local google_drive = {}
local act = require "actions"
local u = require "utils"

google_drive.ctx = nil

local defaultExpiry = 30*1000*1000*1000
-- db Expiry Timer for Interrupted uploads 7 days.
local resUplExpiry  = 7*24*60*60*1000*1000*1000 

-- Request types
local EMBEDDED="embedded"
local BATCH="batch"

-- States
-- Waiting for http request
local HTTP = "http"
-- Waiting for file name
local FNAME = "fname"
-- Waiting for body
local BODY = "body"

-- Google Drive upload protocol:
-- Google drive differentiates between new upload and update to existing
-- doc. Any new upload uses POST and update uses PUT.
-- New objects seem to use mime-multipart with embedded objects
-- Update seem to use two phase upload with file going native.
-- Folder upload can use x-goog-upload-protocol:batch header
-- In this case multiple files are uploaded in a single put
-- as mime-multipart. Need a clean way of handling it that does not require
-- full body read
function google_drive.upload_request(req)
   local url = req.url
   
   google_drive.ctx = nil
   
   -- check if it is upload
   if string.match(url, "/upload/drive") ~= nil then
      if req.method == "PUT" or req.method == "POST" then
	 local h = req.headers
	 if h["X-Goog-Upload-Protocol"] == "batch" then
	    google_drive.ctx = {type=BATCH, state=FNAME}
	    act.ignore_request()
	    return   
	 end
	 
	 local query = req.query	 
	 -- Check if it multipart upload
	 if query["uploadType"] ~= nil and query["uploadType"][1] == "multipart" then
	    -- This is embedded multipart upload type and should be handled by part request
	    -- Note ctx is global
	    google_drive.ctx = {type=EMBEDDED, state=FNAME}
	    act.ignore_request()
	    return
	 end
	 local dbKey
	 if query["upload_id"] == nil then
	    dbKey = "google-drive:upload" .. url .. query["key"][1]
	    -- This is the first request that provides the information
	    local body = req.body
	    if body == nil then
	       return
	    end
	    if body["originalFilename"] ~= nil then
	       db.set(db.Lru, dbKey, body["originalFilename"], defaultExpiry)
	       -- We are done with request there is no point further inspecing data
	       act.ignore_request()
	    elseif body["title"] ~= nil then
	       db.set(db.Lru, dbKey, body["title"], defaultExpiry)
	       -- We are done with request there is no point further inspecing data
	       act.ignore_request()
	    else
	       err("[google_drive]: upload first request: did not find title")
	       u.deep_print(body)
	    end
	    return
	 elseif query["upload_id"] ~= nil then
            -- Subsequent upload request in case of resumable uploads
	    dbKey = "google-drive:upload" .. query["upload_id"][1]
  	 end
	 -- Ignore requests that dont carry the initial part of the file (Content-Range header starting with non 0)
	 -- NTU was raised and the entire content of the file is inspected when the first request is processed (Content-Range header starting with 0)
	 local headers = req.headers
	 local range = headers["Content-Range"]
	 if range ~= nil then
		if not string.match(range, 'bytes 0') then
			act.ignore_request()
			return
		end
	 end
	 -- Actual upload request
	 local fname = db.get(db.Lru, dbKey)
	 if fname ~= nil then
	    act.file_upload(fname)
	    return	
	 else
	    -- act.ignore_request()
	 end
    end	
   end
end

-- Handle incoming requests
function google_drive.request(req)
   local h = req.host
   if string.match(h, "clients%d.google.com") ~= nil then
      return google_drive.upload_request(req)
   elseif string.match(h, "googleusercontent.com") ~= nil then
      -- Download request is completely handled in go
   end
end

function google_drive.request_part(p)
   if google_drive.ctx ~= nil and (google_drive.ctx["type"] == EMBEDDED or google_drive.ctx["type"] == BATCH) then
      if google_drive.ctx["state"] == FNAME then 
         local body = p.body
         if body ~= nil then
            if body["originalFilename"] ~= nil then
               google_drive.ctx["file"] = body["originalFilename"]
            elseif body["title"] ~= nil then
               google_drive.ctx["file"] = body["title"]
            end
            google_drive.ctx["state"] = BODY
            act.ignore_request()
         end
      elseif google_drive.ctx["state"] == BODY then
         act.file_upload(google_drive.ctx["file"])
         google_drive.ctx["state"] = FNAME
      end
   end
end
-- Handle outgoing response
function google_drive.response(resp)
	local res = resp.headers
	if google_drive.ctx == nil then
		return
	end
	if google_drive.ctx["file"] == nil then
		return
	end

	if res ~= nil then
		if res["X-Guploader-Uploadid"] ~= nil then
			local dbKey = "google-drive:upload" .. res["X-Guploader-Uploadid"]
			google_drive.ctx["upload_id"] = res["X-Guploader-Uploadid"]
			db.set(db.Lru, dbKey, google_drive.ctx["file"] , resUplExpiry)
		end
	end
end


return google_drive
