local outlook_email = {}
local act = require "actions"
local u = require "utils"

local defaultExpiry = 10 * 24 * 1000 * 1000 * 1000

--- collect e-mail addresses from To, CC, BCC
function outlook_email.collectEmailAddresses(emails)
   local result = {}
   if emails == nil then
      return result
   end
   for i, e in ipairs(emails) do
      if e["EmailAddress"] ~= nil and e["Name"] ~= nil then
         table.insert(result, e["Name"] .. " <" .. e["EmailAddress"] .. ">")
      elseif e["EmailAddress"] ~= nil then
         table.insert(result, e["EmailAddress"])
      end
   end
   return result
end

--- In some cases the outlook365 uses doclinks for attachments
--- Collect them when you can
function outlook_email.collectDocLinks(docLinks)
   local result = {}
   for i, e in ipairs(docLinks) do
      local link = {provider = e["ProviderType"], url = e["Url"]}
      table.insert(result, link)
   end
   return result
end

--- Collect Attachments sent from your computer
function outlook_email.collectAttachments(attachments)
   local result = {}
   for i, e in ipairs(attachments) do
      table.insert(result, e["Name"])
   end
   return result
end

--- Parse attachment actions
function outlook_email.attachAction(headers)
   local req = decode_url_json(headers["X-Owa-Urlpostdata"])
   if req["Body"] == nil or req["Body"]["Attachments"] == nil or req["Body"]["Attachments"][1] == nil then
      return
   end

   local fname = req["Body"]["Attachments"][1]["Name"]
   if fname ~= nil then
      act.file_upload(fname)
   end
end

--- Handle deletion of attachment
--- Note we don't have the e-mail ID at this point, only the attachment id
function outlook_email.deleteAttachAction(headers)
   local req = decode_url_json(headers["X-Owa-Urlpostdata"])
end

--- Handle attachment referenced from the one drive
function outlook_email.referenceAttachmentAction(headers)
   local req = decode_url_json(headers["X-Owa-Urlpostdata"])
   local location = req["location"]
end

--- Handle attachment referenced from the one drive
function outlook_email.referenceAttachmentAction(actionName, headers, req)
   if actionName == "CreateReferenceAttachmentFromLocalFileActionComputer" then
      if headers["X-Owa-Urlpostdata"] ~= nil then
         local post_data = decode_url_json(headers["X-Owa-Urlpostdata"])
         if post_data ~= nil and post_data.requestObject ~= nil and post_data.requestObject.FileContentToUpload ~= nil then
            local data = decode_base64(post_data.requestObject.FileContentToUpload)
            if data ~= nil then
               act.file_upload("POSTDATA.TXT", data)
            end
         end
      else
         local head = json_embedded_file(req, "requestObject.FileContentToUpload", "base64")
         if head ~= nil then
            act.file_upload("POSTDATA.TXT", head)
         end
      end
   end
end

function outlook_email.createAction(headers, body)
   local draft = false
   if body["Body"]["MessageDisposition"] == "SaveOnly" then
      draft = true
   end

   if body["Body"]["Items"][1] == nil then
      return
   end
   local item = body["Body"]["Items"][1]
   local to = outlook_email.collectEmailAddresses(item["ToRecipients"])
   local cc = outlook_email.collectEmailAddresses(item["CcRecipients"])
   local bcc = outlook_email.collectEmailAddresses(item["BccRecipients"])
   local subject = item["Subject"]
   local from = headers["X-Upnanchormailbox"]
   local b
   local ishtml = true

   if item["Body"] ~= nil then
      b = item["Body"]["Value"]
      ishtml = item["Body"]["BodyType"] == "HTML"
   end
   act.sendMail(draft, from, to, cc, bcc, subject, b, ishtml)
end
-- Handle e-mail send action
function outlook_email.sendAction(headers, body, actionName)
   -- Is it draft?
   local ishtml = true
   local draft = false
   local updates = nil

   if body["Body"] == nil then
      return
   end
   if actionName == "CreateItem" or actionName == "CreateMessageForComposeSend" or actionName == "CreateMessageForCompose" then
      if body["Body"]["Items"] ~= nil then
         outlook_email.createAction(headers, body)
         return
      end
   else
      if body["Body"]["ItemChanges"] ~= nil and body["Body"]["ItemChanges"][1] ~= nil then
         updates = body["Body"]["ItemChanges"][1]["Updates"]
      end
   end
   if updates == nil then
      return
   end
   local from = headers["X-Upnanchormailbox"]

   if body["Body"]["MessageDisposition"] == "SaveOnly" then
      draft = true
   end
   local to, cc, bcc, subject, attachments, b
   for k, v in ipairs(updates) do
      if v["Path"] ~= nil and v["Item"] ~= nil then
         local fieldURI = v["Path"]["FieldURI"]
         local value = v["Item"][fieldURI]

         if (fieldURI == "ToRecipients") then
            to = outlook_email.collectEmailAddresses(value)
         elseif fieldURI == "CcRecipients" then
            cc = outlook_email.collectEmailAddresses(value)
         elseif fieldURI == "BccRecipients" then
            bcc = outlook_email.collectEmailAddresses(value)
         elseif fieldURI == "Subject" then
            subject = value
         elseif fieldURI == "DocLinks" then
            attachments = outlook_email.collectDocLinks(value)
         elseif fieldURI == "Body" then
            b = value["Value"]
            ishtml = value["BodyType"] == "HTML"
         end
      end
   end
   act.sendMail(draft, from, to, cc, bcc, subject, b, ishtml)
end

--- Handle an incoming request from the browser
function outlook_email.request(req)
   local headers = req.headers

   --- Is this Outlook EWS
   local agent = headers["User-Agent"]
   if string.sub(agent, 1, 10) == "MacOutlook" then
      local action = headers["Soapaction"]
      if action ~= nil then
         if action ==  "\"http://schemas.microsoft.com/exchange/services/2006/messages/CreateItem\"" then
            act.ewsMail("CreateItem")
         elseif action == "\"http://schemas.microsoft.com/exchange/services/2006/messages/UpdateItem\"" then
            act.ewsMail("UpdateItem")
         end
         --- All Soapactions processing end here
         return
      end
   end

   local actionName = headers["X-Owa-Actionname"]
   if actionName == nil then
      actionName = headers["X-Owa-Action"]
   end
   if actionName == nil then
      actionName = headers["Action"]
   end

   local sendActions = {
      "CreateMessageForComposeSend",
      "CreateMessageForCompose",
      "UpdateMessageForComposeSend",
      "CreateItem",
      "UpdateItem",
      "UpdateMessageForCompose",
      "CreateResponse",
      "MailComposeUpgrade"
   }
   local deleteAttachmentActions = {"DeleteAttachmentAction", "DeleteAttachment"}

   if actionName == nil then
      return
   elseif actionName == "CreateAttachmentFromLocalFile" or actionName == "P_CreateAttachmentFromLocalFile" then
      outlook_email.attachAction(headers)
   elseif u.has_value(actionName, deleteAttachmentActions) then
      outlook_email.deleteAttachAction(headers)
   elseif u.has_value(actionName, sendActions) then
      local body = req.body
      if body ~= nil then
         outlook_email.sendAction(headers, body, actionName)
      end
   elseif actionName == "CreateReferenceAttachmentFromLocalFileActionComputer" then
      outlook_email.referenceAttachmentAction(actionName, headers, req)
   elseif actionName == "ComposeForms" then
   elseif actionName == "LogDatapointAction" then
   elseif actionName == "CreateReferenceAttachmentFromAttachmentDataProviderActionMy files" then
      outlook_email.referenceAttachmentAction(headers)
   else
      return
   end
end

-- Handle response to the browser
function outlook_email.response(resp)
end

return outlook_email
