// Copyright 2016-2018, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package backend

import (
	"context"
	"encoding/json"

	"github.com/pulumi/pulumi/sdk/v3/go/common/resource/plugin"
	"github.com/pulumi/pulumi/sdk/v3/go/common/util/result"
	"github.com/pulumi/pulumi/sdk/v3/go/common/workspace"
)

// PublishOperation publishes a PolicyPack to the backend.
type PublishOperation struct {
	Root       string
	PlugCtx    *plugin.Context
	PolicyPack *workspace.PolicyPackProject
	Scopes     CancellationScopeSource
}

// PolicyPackOperation is used to make various operations against a Policy Pack.
type PolicyPackOperation struct {
	// If nil, the latest version is assumed.
	VersionTag *string
	Scopes     CancellationScopeSource
	Config     map[string]*json.RawMessage
}

// PolicyPack is used to manage policy against a pluggable backend.
type PolicyPack interface {
	// Ref returns a reference to this PolicyPack.
	Ref() PolicyPackReference
	// Backend returns the backend this PolicyPack is managed by.
	Backend() Backend
	// Publish the PolicyPack to the service.
	Publish(ctx context.Context, op PublishOperation) result.Result
	// Enable the PolicyPack to a Policy Group in an organization. If Policy Group is
	// empty, it enables it for the default Policy Group.
	Enable(ctx context.Context, policyGroup string, op PolicyPackOperation) error

	// Disable the PolicyPack for a Policy Group in an organization. If Policy Group is
	// empty, it disables it for the default Policy Group.
	Disable(ctx context.Context, policyGroup string, op PolicyPackOperation) error

	// Validate the PolicyPack configuration against configuration schema.
	Validate(ctx context.Context, op PolicyPackOperation) error

	// Remove the PolicyPack from an organization. The Policy Pack must be removed from
	// all Policy Groups before it can be removed.
	Remove(ctx context.Context, op PolicyPackOperation) error
}
