// Copyright 2016-2020, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package dotnet

import (
	"encoding/json"

	"github.com/pulumi/pulumi/pkg/v3/codegen/schema"
)

// CSharpPropertyInfo represents the C# language-specific info for a property.
type CSharpPropertyInfo struct {
	Name string `json:"name,omitempty"`
}

// CSharpPackageInfo represents the C# language-specific info for a package.
type CSharpPackageInfo struct {
	PackageReferences      map[string]string `json:"packageReferences,omitempty"`
	Namespaces             map[string]string `json:"namespaces,omitempty"`
	Compatibility          string            `json:"compatibility,omitempty"`
	DictionaryConstructors bool              `json:"dictionaryConstructors,omitempty"`
	ProjectReferences      []string          `json:"projectReferences,omitempty"`
	// Determines whether to make single-return-value methods return an output object or the single value.
	LiftSingleValueMethodReturns bool `json:"liftSingleValueMethodReturns,omitempty"`

	// The root namespace used for the package. This defaults to `Pulumi`.
	RootNamespace string `json:"rootNamespace,omitempty"`

	// Allow the Pkg.Version field to filter down to emitted code.
	RespectSchemaVersion bool `json:"respectSchemaVersion,omitempty"`
}

// Returns the root namespace, or "Pulumi" if not provided.
func (info *CSharpPackageInfo) GetRootNamespace() string {
	if r := info.RootNamespace; r != "" {
		return r
	}
	return "Pulumi"
}

// Importer implements schema.Language for .NET.
var Importer schema.Language = importer(0)

type importer int

// ImportDefaultSpec decodes language-specific metadata associated with a DefaultValue.
func (importer) ImportDefaultSpec(def *schema.DefaultValue, raw json.RawMessage) (interface{}, error) {
	return raw, nil
}

// ImportPropertySpec decodes language-specific metadata associated with a Property.
func (importer) ImportPropertySpec(property *schema.Property, raw json.RawMessage) (interface{}, error) {
	var info CSharpPropertyInfo
	if err := json.Unmarshal([]byte(raw), &info); err != nil {
		return nil, err
	}
	return info, nil
}

// ImportObjectTypeSpec decodes language-specific metadata associated with a ObjectType.
func (importer) ImportObjectTypeSpec(object *schema.ObjectType, raw json.RawMessage) (interface{}, error) {
	return raw, nil
}

// ImportResourceSpec decodes language-specific metadata associated with a Resource.
func (importer) ImportResourceSpec(resource *schema.Resource, raw json.RawMessage) (interface{}, error) {
	return raw, nil
}

// ImportFunctionSpec decodes language-specific metadata associated with a Function.
func (importer) ImportFunctionSpec(function *schema.Function, raw json.RawMessage) (interface{}, error) {
	return raw, nil
}

// ImportPackageSpec decodes language-specific metadata associated with a Package.
func (importer) ImportPackageSpec(pkg *schema.Package, raw json.RawMessage) (interface{}, error) {
	var info CSharpPackageInfo
	if err := json.Unmarshal([]byte(raw), &info); err != nil {
		return nil, err
	}
	return info, nil
}
