// Copyright 2016-2022, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// When GeneratePackage generates TypeScript files, there is a need to track
// internally which definitions are emitted into these files to
// re-export them efficiently. This file defines helper structs for
// this information. The tracking is approximate.
package nodejs

// Packages information about a TypeScript file generated by
// GeneratePackage.
type fileInfo struct {
	fileType         fileType
	pathToNodeModule string           // path understood by require() in Node
	resourceFileInfo resourceFileInfo // required if fileType == resourceFileType
	functionFileInfo functionFileInfo // required if fileType == functionFileType
}

type fileType int

const (
	resourceFileType fileType = iota // files that define Pulumi Resources
	functionFileType                 // files that define Pulumi functions
	otherFileType                    // everything else
)

type resourceFileInfo struct {
	resourceClassName         string
	resourceArgsInterfaceName string
	stateInterfaceName        string // may be empty
	methodsNamespaceName      string // may be empty
}

// TypeScript interface names exported from the file.
func (ri resourceFileInfo) interfaces() []string {
	return nonEmptyStrings([]string{
		ri.resourceArgsInterfaceName,
		ri.stateInterfaceName,
	})
}

type functionFileInfo struct {
	functionName                           string
	functionArgsInterfaceName              string // may be empty
	functionResultInterfaceName            string // may be empty
	functionOutputVersionName              string // may be empty
	functionOutputVersionArgsInterfaceName string // may be empty
}

// TypeScript function names exported from the file.
func (fi functionFileInfo) functions() []string {
	return nonEmptyStrings([]string{
		fi.functionName,
		fi.functionOutputVersionName,
	})
}

// TypeScript interface names exported from the file.
func (fi functionFileInfo) interfaces() []string {
	return nonEmptyStrings([]string{
		fi.functionArgsInterfaceName,
		fi.functionResultInterfaceName,
		fi.functionOutputVersionArgsInterfaceName,
	})
}
