// Copyright 2016-2020, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package pcl

import (
	"github.com/hashicorp/hcl/v2"
	"github.com/hashicorp/hcl/v2/hclsyntax"
	"github.com/pulumi/pulumi/pkg/v3/codegen/hcl2/model"
)

// ConfigVariable represents a program- or component-scoped input variable. The value for a config variable may come
// from stack configuration or component inputs, respectively, and may have a default value.
type ConfigVariable struct {
	node

	syntax *hclsyntax.Block
	typ    model.Type

	// The variable definition.
	Definition *model.Block
	// The default value for the config variable, if any.
	DefaultValue model.Expression

	// The name visible to API calls related to the config. Used as the argument when
	// fetching config variables. Must not be modified during code generation to ensure
	// that valid config calls don't become invalid.
	logicalName string
}

// SyntaxNode returns the syntax node associated with the config variable.
func (cv *ConfigVariable) SyntaxNode() hclsyntax.Node {
	return cv.syntax
}

func (cv *ConfigVariable) Traverse(traverser hcl.Traverser) (model.Traversable, hcl.Diagnostics) {
	return cv.typ.Traverse(traverser)
}

func (cv *ConfigVariable) VisitExpressions(pre, post model.ExpressionVisitor) hcl.Diagnostics {
	return model.VisitExpressions(cv.Definition, pre, post)
}

func (cv *ConfigVariable) Name() string {
	return cv.Definition.Labels[0]
}

func (cv *ConfigVariable) LogicalName() string {
	if cv.logicalName != "" {
		return cv.logicalName
	}
	return cv.Name()
}

// Type returns the type of the config variable.
func (cv *ConfigVariable) Type() model.Type {
	return cv.typ
}
