// Copyright 2016-2018, Pulumi Corporation.  All rights reserved.
//go:build (nodejs || all) && !smoke

package ints

import (
	"testing"

	"github.com/stretchr/testify/assert"

	"github.com/pulumi/pulumi/pkg/v3/resource/deploy/providers"
	"github.com/pulumi/pulumi/pkg/v3/testing/integration"
	"github.com/pulumi/pulumi/sdk/v3/go/common/resource"
)

// TestProtectedResources tests some interesting operations on protected resources.
func TestProtectedResources(t *testing.T) {
	integration.ProgramTest(t, &integration.ProgramTestOptions{
		Dir:          "step1",
		Dependencies: []string{"@pulumi/pulumi"},
		Quick:        true,
		ExtraRuntimeValidation: func(t *testing.T, stackInfo integration.RuntimeValidationStackInfo) {
			// A single synthetic stack and a single "eternal" resource.
			assert.NotNil(t, stackInfo.Deployment)
			assert.Equal(t, 3, len(stackInfo.Deployment.Resources))
			stackRes := stackInfo.Deployment.Resources[0]
			assert.Equal(t, resource.RootStackType, stackRes.URN.Type())
			providerRes := stackInfo.Deployment.Resources[1]
			assert.True(t, providers.IsProviderType(providerRes.URN.Type()))
			a := stackInfo.Deployment.Resources[2]
			assert.Equal(t, "eternal", string(a.URN.Name()))
			assert.True(t, a.Protect)
		},
		EditDirs: []integration.EditDir{
			{
				Dir:      "step2",
				Additive: true,
				ExtraRuntimeValidation: func(t *testing.T, stackInfo integration.RuntimeValidationStackInfo) {
					// An update to "eternal"; should still be there.
					assert.NotNil(t, stackInfo.Deployment)
					assert.Equal(t, 3, len(stackInfo.Deployment.Resources))
					stackRes := stackInfo.Deployment.Resources[0]
					assert.Equal(t, resource.RootStackType, stackRes.URN.Type())
					providerRes := stackInfo.Deployment.Resources[1]
					assert.True(t, providers.IsProviderType(providerRes.URN.Type()))
					a := stackInfo.Deployment.Resources[2]
					assert.Equal(t, "eternal", string(a.URN.Name()))
					assert.True(t, a.Protect)
				},
			},
			{
				Dir:      "step3",
				Additive: true,
				// This step will fail because the resource is protected.
				ExpectFailure: true,
				ExtraRuntimeValidation: func(t *testing.T, stackInfo integration.RuntimeValidationStackInfo) {
					// The protected resource should still be in the snapshot and it should still be protected.
					assert.NotNil(t, stackInfo.Deployment)
					assert.Equal(t, 3, len(stackInfo.Deployment.Resources))
					stackRes := stackInfo.Deployment.Resources[0]
					assert.Equal(t, resource.RootStackType, stackRes.URN.Type())
					providerRes := stackInfo.Deployment.Resources[1]
					assert.True(t, providers.IsProviderType(providerRes.URN.Type()))
					a := stackInfo.Deployment.Resources[2]
					assert.Equal(t, "eternal", string(a.URN.Name()))
					assert.True(t, a.Protect)
				},
			},
			{
				Dir:      "step4",
				Additive: true,
				ExtraRuntimeValidation: func(t *testing.T, stackInfo integration.RuntimeValidationStackInfo) {
					// "eternal" should now be unprotected.
					assert.NotNil(t, stackInfo.Deployment)
					assert.Equal(t, 3, len(stackInfo.Deployment.Resources))
					stackRes := stackInfo.Deployment.Resources[0]
					assert.Equal(t, resource.RootStackType, stackRes.URN.Type())
					providerRes := stackInfo.Deployment.Resources[1]
					assert.True(t, providers.IsProviderType(providerRes.URN.Type()))
					a := stackInfo.Deployment.Resources[2]
					assert.Equal(t, "eternal", string(a.URN.Name()))
					assert.False(t, a.Protect)
				},
			},
			{
				Dir:      "step5",
				Additive: true,
				ExtraRuntimeValidation: func(t *testing.T, stackInfo integration.RuntimeValidationStackInfo) {
					// Finally, "eternal" should be deleted.
					assert.NotNil(t, stackInfo.Deployment)
					assert.Equal(t, 1, len(stackInfo.Deployment.Resources))
					stackRes := stackInfo.Deployment.Resources[0]
					assert.Equal(t, resource.RootStackType, stackRes.URN.Type())
				},
			},
		},
	})
}
