// Copyright 2016-2018, Pulumi Corporation.  All rights reserved.
//go:build !smoke

package ints

import (
	"testing"

	"github.com/pulumi/pulumi/pkg/v3/testing/integration"
	"github.com/pulumi/pulumi/sdk/v3/go/common/resource"
	"github.com/pulumi/pulumi/sdk/v3/go/common/tokens"
	"github.com/stretchr/testify/assert"
)

var Dirs = []string{
	"simple",
}

func Validator(language string) func(t *testing.T, stack integration.RuntimeValidationStackInfo) {
	dynamicResName := "pulumi-" + language + ":dynamic:Resource"
	return func(t *testing.T, stack integration.RuntimeValidationStackInfo) {
		foundRes1 := false
		foundRes2Child := false
		foundRes3 := false
		foundRes4Child := false
		foundRes5Child := false
		for _, res := range stack.Deployment.Resources {
			// "res1" has a transformation which adds additionalSecretOutputs
			if res.URN.Name() == "res1" {
				foundRes1 = true
				assert.Equal(t, res.Type, tokens.Type(dynamicResName))
				assert.Contains(t, res.AdditionalSecretOutputs, resource.PropertyKey("output"))
			}
			// "res2" has a transformation which adds additionalSecretOutputs to it's
			// "child"
			if res.URN.Name() == "res2-child" {
				foundRes2Child = true
				assert.Equal(t, res.Type, tokens.Type(dynamicResName))
				assert.Equal(t, res.Parent.Type(), tokens.Type("my:component:MyComponent"))
				assert.Contains(t, res.AdditionalSecretOutputs, resource.PropertyKey("output"))
				assert.Contains(t, res.AdditionalSecretOutputs, resource.PropertyKey("output2"))
			}
			// "res3" is impacted by a global stack transformation which sets
			// optionalDefault to "stackDefault"
			if res.URN.Name() == "res3" {
				foundRes3 = true
				assert.Equal(t, res.Type, tokens.Type(dynamicResName))
				optionalInput := res.Inputs["optionalInput"]
				assert.NotNil(t, optionalInput)
				assert.Equal(t, "stackDefault", optionalInput.(string))
			}
			// "res4" is impacted by two component parent transformations which set
			// optionalDefault to "default1" and then "default2" and also a global stack
			// transformation which sets optionalDefault to "stackDefault".  The end
			// result should be "stackDefault".
			if res.URN.Name() == "res4-child" {
				foundRes4Child = true
				assert.Equal(t, res.Type, tokens.Type(dynamicResName))
				assert.Equal(t, res.Parent.Type(), tokens.Type("my:component:MyComponent"))
				optionalInput := res.Inputs["optionalInput"]
				assert.NotNil(t, optionalInput)
				assert.Equal(t, "stackDefault", optionalInput.(string))
			}
			// "res5" modifies one of its children to depend on another of its children.
			if res.URN.Name() == "res5-child1" {
				foundRes5Child = true
				assert.Equal(t, res.Type, tokens.Type(dynamicResName))
				assert.Equal(t, res.Parent.Type(), tokens.Type("my:component:MyComponent"))
				// TODO[pulumi/pulumi#3282] Due to this bug, the dependency information
				// will not be correctly recorded in the state file, and so cannot be
				// verified here.
				//
				// assert.Len(t, res.PropertyDependencies, 1)
				input := res.Inputs["input"]
				assert.NotNil(t, input)
				assert.Equal(t, "b", input.(string))
			}
		}
		assert.True(t, foundRes1)
		assert.True(t, foundRes2Child)
		assert.True(t, foundRes3)
		assert.True(t, foundRes4Child)
		assert.True(t, foundRes5Child)
	}
}
