// Abstract contract for the full ERC 20 Token standard

// https://github.com/ethereum/EIPs/issues/20

pragma solidity ^0.4.8;



contract Token {

    /* This is a slight change to the ERC20 base standard.

    function totalSupply() constant returns (uint256 supply);

    is replaced with:

    uint256 public totalSupply;

    This automatically creates a getter function for the totalSupply.

    This is moved to the base contract since public getter functions are not

    currently recognised as an implementation of the matching abstract

    function by the compiler.

    */

    /// total amount of tokens

    uint256 public totalSupply;



    /// @param _owner The address from which the balance will be retrieved

    /// @return The balance

    function balanceOf(address _owner) constant returns (uint256 balance);



    /// @notice send `_value` token to `_to` from `msg.sender`

    /// @param _to The address of the recipient

    /// @param _value The amount of token to be transferred

    /// @return Whether the transfer was successful or not

    function transfer(address _to, uint256 _value) returns (bool success);



    /// @notice send `_value` token to `_to` from `_from` on the condition it is approved by `_from`

    /// @param _from The address of the sender

    /// @param _to The address of the recipient

    /// @param _value The amount of token to be transferred

    /// @return Whether the transfer was successful or not

    function transferFrom(address _from, address _to, uint256 _value) returns (bool success);



    /// @notice `msg.sender` approves `_spender` to spend `_value` tokens

    /// @param _spender The address of the account able to transfer the tokens

    /// @param _value The amount of tokens to be approved for transfer

    /// @return Whether the approval was successful or not

    function approve(address _spender, uint256 _value) returns (bool success);



    /// @param _owner The address of the account owning tokens

    /// @param _spender The address of the account able to transfer the tokens

    /// @return Amount of remaining tokens allowed to spent

    function allowance(address _owner, address _spender) constant returns (uint256 remaining);



    event Transfer(address indexed _from, address indexed _to, uint256 _value);

    event Approval(address indexed _owner, address indexed _spender, uint256 _value);

}





contract StandardToken is Token {



    function transfer(address _to, uint256 _value) returns (bool success) {

        //Default assumes totalSupply can't be over max (2^256 - 1).

        //If your token leaves out totalSupply and can issue more tokens as time goes on, you need to check if it doesn't wrap.

        //Replace the if with this one instead.

        if (balances[msg.sender] >= _value && balances[_to] + _value > balances[_to]) {

            balances[msg.sender] -= _value;

            balances[_to] += _value;

            Transfer(msg.sender, _to, _value);

            return true;

        } else { return false; }

    }



    function transferFrom(address _from, address _to, uint256 _value) returns (bool success) {

        //same as above. Replace this line with the following if you want to protect against wrapping uints.

        if (balances[_from] >= _value && allowed[_from][msg.sender] >= _value && balances[_to] + _value > balances[_to]) {

            balances[_to] += _value;

            balances[_from] -= _value;

            allowed[_from][msg.sender] -= _value;

            Transfer(_from, _to, _value);

            return true;

        } else { return false; }

    }



    function balanceOf(address _owner) constant returns (uint256 balance) {

        return balances[_owner];

    }



    function approve(address _spender, uint256 _value) returns (bool success) {

        allowed[msg.sender][_spender] = _value;

        Approval(msg.sender, _spender, _value);

        return true;

    }



    function allowance(address _owner, address _spender) constant returns (uint256 remaining) {

      return allowed[_owner][_spender];

    }



    mapping (address => uint256) balances;

    mapping (address => mapping (address => uint256)) allowed;

}



contract UGToken is StandardToken {



    function () {

        //if ether is sent to this address, send it back.

        throw;

    }



    string public name = "UG Token";                   //fancy name: eg Simon Bucks

    uint8 public decimals = 18;                //How many decimals to show. ie. There could 1000 base units with 3 decimals. Meaning 0.980 SBX = 980 base units. It's like comparing 1 wei to 1 ether.

    string public symbol = "UGT";                 //An identifier: eg SBX

    string public version = 'v0.1';       //ug 0.1 standard. Just an arbitrary versioning scheme.



    address public founder; // The address of the founder

    uint256 public allocateStartBlock; // The start block number that starts to allocate token to users.

    uint256 public allocateEndBlock; // The end block nubmer that allocate token to users, lasted for a week.



    // The nonce for avoid transfer replay attacks

    mapping(address => uint256) nonces;



    function UGToken() {

        founder = msg.sender;

        allocateStartBlock = block.number;

        allocateEndBlock = allocateStartBlock + 5082; // about one day

    }



    /*

     * Proxy transfer ug token. When some users of the ethereum account has no ether,

     * he or she can authorize the agent for broadcast transactions, and agents may charge agency fees

     * @param _from

     * @param _to

     * @param _value

     * @param feeUgt

     * @param _v

     * @param _r

     * @param _s

     */

    function transferProxy(address _from, address _to, uint256 _value, uint256 _feeUgt,

        uint8 _v,bytes32 _r, bytes32 _s) returns (bool){



        if(balances[_from] < _feeUgt + _value) throw;



        uint256 nonce = nonces[_from];

        bytes32 h = sha3(_from,_to,_value,_feeUgt,nonce);

        if(_from != ecrecover(h,_v,_r,_s)) throw;



        if(balances[_to] + _value < balances[_to]

            || balances[msg.sender] + _feeUgt < balances[msg.sender]) throw;

        balances[_to] += _value;

        Transfer(_from, _to, _value);



        balances[msg.sender] += _feeUgt;

        Transfer(_from, msg.sender, _feeUgt);



        balances[_from] -= _value + _feeUgt;

        nonces[_from] = nonce + 1;

        return true;

    }



    /*

     * Proxy approve that some one can authorize the agent for broadcast transaction

     * which call approve method, and agents may charge agency fees

     * @param _from The  address which should tranfer ugt to others

     * @param _spender The spender who allowed by _from

     * @param _value The value that should be tranfered.

     * @param _v

     * @param _r

     * @param _s

     */

    function approveProxy(address _from, address _spender, uint256 _value,

        uint8 _v,bytes32 _r, bytes32 _s) returns (bool success) {



        uint256 nonce = nonces[_from];

        bytes32 hash = sha3(_from,_spender,_value,nonce);

        if(_from != ecrecover(hash,_v,_r,_s)) throw;

        allowed[_from][_spender] = _value;

        Approval(_from, _spender, _value);

        nonces[_from] = nonce + 1;

        return true;

    }





    /*

     * Get the nonce

     * @param _addr

     */

    function getNonce(address _addr) constant returns (uint256){

        return nonces[_addr];

    }



    /* Approves and then calls the receiving contract */

    function approveAndCall(address _spender, uint256 _value, bytes _extraData) returns (bool success) {

        allowed[msg.sender][_spender] = _value;

        Approval(msg.sender, _spender, _value);



        //call the receiveApproval function on the contract you want to be notified. This crafts the function signature manually so one doesn't have to include a contract in here just for this.

        //receiveApproval(address _from, uint256 _value, address _tokenContract, bytes _extraData)

        //it is assumed that when does this that the call *should* succeed, otherwise one would use vanilla approve instead.

        if(!_spender.call(bytes4(bytes32(sha3("receiveApproval(address,uint256,address,bytes)"))), msg.sender, _value, this, _extraData)) { throw; }

        return true;

    }



    /* Approves and then calls the contract code*/

    function approveAndCallcode(address _spender, uint256 _value, bytes _extraData) returns (bool success) {

        allowed[msg.sender][_spender] = _value;

        Approval(msg.sender, _spender, _value);



        //Call the contract code

        if(!_spender.call(_extraData)) { throw; }

        return true;

    }



    // Allocate tokens to the users

    // @param _owners The owners list of the token

    // @param _values The value list of the token

    function allocateTokens(address[] _owners, uint256[] _values) {



        if(msg.sender != founder) throw;

        if(block.number < allocateStartBlock || block.number > allocateEndBlock) throw;

        if(_owners.length != _values.length) throw;



        for(uint256 i = 0; i < _owners.length ; i++){

            address owner = _owners[i];

            uint256 value = _values[i];

            if(totalSupply + value <= totalSupply || balances[owner] + value <= balances[owner]) throw;

            totalSupply += value;

            balances[owner] += value;

        }

    }

}