pragma solidity ^0.4.11;





/**

 * @title SafeMath

 * @dev Math operations with safety checks that throw on error

 */

library SafeMath {

  function mul(uint256 a, uint256 b) internal constant returns (uint256) {

    uint256 c = a * b;

    assert(a == 0 || c / a == b);

    return c;

  }



  function div(uint256 a, uint256 b) internal constant returns (uint256) {

    // assert(b > 0); // Solidity automatically throws when dividing by 0

    uint256 c = a / b;

    // assert(a == b * c + a % b); // There is no case in which this doesn't hold

    return c;

  }



  function sub(uint256 a, uint256 b) internal constant returns (uint256) {

    assert(b <= a);

    return a - b;

  }



  function add(uint256 a, uint256 b) internal constant returns (uint256) {

    uint256 c = a + b;

    assert(c >= a);

    return c;

  }

}







/**

 * @title Ownable

 * @dev The Ownable contract has an owner address, and provides basic authorization control

 * functions, this simplifies the implementation of "user permissions".

 */

contract Ownable {

  address public owner;





  /**

   * @dev The Ownable constructor sets the original `owner` of the contract to the sender

   * account.

   */

  function Ownable() {

    owner = msg.sender;

  }





  /**

   * @dev Throws if called by any account other than the owner.

   */

  modifier onlyOwner() {

    require(msg.sender == owner);

    _;

  }





  /**

   * @dev Allows the current owner to transfer control of the contract to a newOwner.

   * @param newOwner The address to transfer ownership to.

   */

  function transferOwnership(address newOwner) onlyOwner {

    require(newOwner != address(0));      

    owner = newOwner;

  }



}





contract Crowdsale {

  using SafeMath for uint256;



  // The token being sold

  MintableToken public token;



  // start and end timestamps where investments are allowed (both inclusive)

  uint256 public startTime;

  uint256 public endTime;



  // address where funds are collected

  address public wallet;



  // how many token units a buyer gets per wei

  uint256 public rate;



  // amount of raised money in wei

  uint256 public weiRaised;



  /**

   * event for token purchase logging

   * @param purchaser who paid for the tokens

   * @param beneficiary who got the tokens

   * @param value weis paid for purchase

   * @param amount amount of tokens purchased

   */ 

  event TokenPurchase(address indexed purchaser, address indexed beneficiary, uint256 value, uint256 amount);





  function Crowdsale(uint256 _startTime, uint256 _endTime, uint256 _rate, address _wallet) {

    require(_startTime >= now);

    require(_endTime >= _startTime);

    require(_rate > 0);

    require(_wallet != 0x0);



    token = createTokenContract();

    startTime = _startTime;

    endTime = _endTime;

    rate = _rate;

    wallet = _wallet;

  }



  // creates the token to be sold. 

  // override this method to have crowdsale of a specific mintable token.

  function createTokenContract() internal returns (MintableToken) {

    return new MintableToken();

  }





  // fallback function can be used to buy tokens

  function () payable {

    buyTokens(msg.sender);

  }



  // low level token purchase function

  function buyTokens(address beneficiary) payable {

    require(beneficiary != 0x0);

    require(validPurchase());



    uint256 weiAmount = msg.value;



    // calculate token amount to be created

    uint256 tokens = weiAmount.mul(rate);



    // update state

    weiRaised = weiRaised.add(weiAmount);



    token.mint(beneficiary, tokens);

    TokenPurchase(msg.sender, beneficiary, weiAmount, tokens);



    forwardFunds();

  }



  // send ether to the fund collection wallet

  // override to create custom fund forwarding mechanisms

  function forwardFunds() internal {

    wallet.transfer(msg.value);

  }



  // @return true if the transaction can buy tokens

  function validPurchase() internal constant returns (bool) {

    bool withinPeriod = now >= startTime && now <= endTime;

    bool nonZeroPurchase = msg.value != 0;

    return withinPeriod && nonZeroPurchase;

  }



  // @return true if crowdsale event has ended

  function hasEnded() public constant returns (bool) {

    return now > endTime;

  }





}





contract CappedCrowdsale is Crowdsale {

  using SafeMath for uint256;



  uint256 public cap;



  function CappedCrowdsale(uint256 _cap) {

    require(_cap > 0);

    cap = _cap;

  }



  // overriding Crowdsale#validPurchase to add extra cap logic

  // @return true if investors can buy at the moment

  function validPurchase() internal constant returns (bool) {

    bool withinCap = weiRaised.add(msg.value) <= cap;

    return super.validPurchase() && withinCap;

  }



  // overriding Crowdsale#hasEnded to add cap logic

  // @return true if crowdsale event has ended

  function hasEnded() public constant returns (bool) {

    bool capReached = weiRaised >= cap;

    return super.hasEnded() || capReached;

  }



}



contract WithdrawVault is Ownable {

    using SafeMath for uint256;



    mapping (address => uint256) public deposited;

    address public wallet;





    function WithdrawVault(address _wallet) {

        require(_wallet != 0x0);

        wallet = _wallet;

    }



    function deposit(address investor) onlyOwner payable {

        deposited[investor] = deposited[investor].add(msg.value);

    }



    function close() onlyOwner {

        wallet.transfer(this.balance);

    }



}









contract ERC20Basic {

  uint256 public totalSupply;

  function balanceOf(address who) constant returns (uint256);

  function transfer(address to, uint256 value) returns (bool);

  event Transfer(address indexed from, address indexed to, uint256 value);

}



contract ERC20 is ERC20Basic {

  function allowance(address owner, address spender) constant returns (uint256);

  function transferFrom(address from, address to, uint256 value) returns (bool);

  function approve(address spender, uint256 value) returns (bool);

  event Approval(address indexed owner, address indexed spender, uint256 value);

}



contract BasicToken is ERC20Basic {

  using SafeMath for uint256;



  mapping(address => uint256) balances;



  /**

  * @dev transfer token for a specified address

  * @param _to The address to transfer to.

  * @param _value The amount to be transferred.

  */

  function transfer(address _to, uint256 _value) returns (bool) {

    require(_to != address(0));



    // SafeMath.sub will throw if there is not enough balance.

    balances[msg.sender] = balances[msg.sender].sub(_value);

    balances[_to] = balances[_to].add(_value);

    Transfer(msg.sender, _to, _value);

    return true;

  }



  /**

  * @dev Gets the balance of the specified address.

  * @param _owner The address to query the the balance of. 

  * @return An uint256 representing the amount owned by the passed address.

  */

  function balanceOf(address _owner) constant returns (uint256 balance) {

    return balances[_owner];

  }



}



contract StandardToken is ERC20, BasicToken {



  mapping (address => mapping (address => uint256)) allowed;





  /**

   * @dev Transfer tokens from one address to another

   * @param _from address The address which you want to send tokens from

   * @param _to address The address which you want to transfer to

   * @param _value uint256 the amount of tokens to be transferred

   */

  function transferFrom(address _from, address _to, uint256 _value) returns (bool) {

    require(_to != address(0));



    var _allowance = allowed[_from][msg.sender];



    // Check is not needed because sub(_allowance, _value) will already throw if this condition is not met

    // require (_value <= _allowance);



    balances[_from] = balances[_from].sub(_value);

    balances[_to] = balances[_to].add(_value);

    allowed[_from][msg.sender] = _allowance.sub(_value);

    Transfer(_from, _to, _value);

    return true;

  }



  /**

   * @dev Approve the passed address to spend the specified amount of tokens on behalf of msg.sender.

   * @param _spender The address which will spend the funds.

   * @param _value The amount of tokens to be spent.

   */

  function approve(address _spender, uint256 _value) returns (bool) {



    // To change the approve amount you first have to reduce the addresses`

    //  allowance to zero by calling `approve(_spender, 0)` if it is not

    //  already 0 to mitigate the race condition described here:

    //  https://github.com/ethereum/EIPs/issues/20#issuecomment-263524729

    require((_value == 0) || (allowed[msg.sender][_spender] == 0));



    allowed[msg.sender][_spender] = _value;

    Approval(msg.sender, _spender, _value);

    return true;

  }



  /**

   * @dev Function to check the amount of tokens that an owner allowed to a spender.

   * @param _owner address The address which owns the funds.

   * @param _spender address The address which will spend the funds.

   * @return A uint256 specifying the amount of tokens still available for the spender.

   */

  function allowance(address _owner, address _spender) constant returns (uint256 remaining) {

    return allowed[_owner][_spender];

  }

  

  /**

   * approve should be called when allowed[_spender] == 0. To increment

   * allowed value is better to use this function to avoid 2 calls (and wait until 

   * the first transaction is mined)

   * From MonolithDAO Token.sol

   */

  function increaseApproval (address _spender, uint _addedValue) 

    returns (bool success) {

    allowed[msg.sender][_spender] = allowed[msg.sender][_spender].add(_addedValue);

    Approval(msg.sender, _spender, allowed[msg.sender][_spender]);

    return true;

  }



  function decreaseApproval (address _spender, uint _subtractedValue) 

    returns (bool success) {

    uint oldValue = allowed[msg.sender][_spender];

    if (_subtractedValue > oldValue) {

      allowed[msg.sender][_spender] = 0;

    } else {

      allowed[msg.sender][_spender] = oldValue.sub(_subtractedValue);

    }

    Approval(msg.sender, _spender, allowed[msg.sender][_spender]);

    return true;

  }



}



contract BurnableToken is StandardToken {



    /**

     * @dev Burns a specific amount of tokens.

     * @param _value The amount of token to be burned.

     */

    function burn(uint _value)

        public

    {

        require(_value > 0);



        address burner = msg.sender;

        balances[burner] = balances[burner].sub(_value);

        totalSupply = totalSupply.sub(_value);

        Burn(burner, _value);

    }



    event Burn(address indexed burner, uint indexed value);

}



contract MintableToken is StandardToken, Ownable {

  event Mint(address indexed to, uint256 amount);

  event MintFinished();



  bool public mintingFinished = false;





  modifier canMint() {

    require(!mintingFinished);

    _;

  }



  /**

   * @dev Function to mint tokens

   * @param _to The address that will receive the minted tokens.

   * @param _amount The amount of tokens to mint.

   * @return A boolean that indicates if the operation was successful.

   */

  function mint(address _to, uint256 _amount) onlyOwner canMint returns (bool) {

    totalSupply = totalSupply.add(_amount);

    balances[_to] = balances[_to].add(_amount);

    Mint(_to, _amount);

    Transfer(0x0, _to, _amount);

    return true;

  }



  /**

   * @dev Function to stop minting new tokens.

   * @return True if the operation was successful.

   */

  function finishMinting() onlyOwner returns (bool) {

    mintingFinished = true;

    MintFinished();

    return true;

  }

}



















contract FinalizableCrowdsale is Crowdsale, Ownable {

  using SafeMath for uint256;



  bool public isFinalized = false;



  event Finalized();



  /**

   * @dev Must be called after crowdsale ends, to do some extra finalization

   * work. Calls the contract's finalization function.

   */

  function finalize() onlyOwner {

    require(!isFinalized);

    require(hasEnded());



    finalization();

    Finalized();

    

    isFinalized = true;

  }



  /**

   * @dev Can be overriden to add finalization logic. The overriding function

   * should call super.finalization() to ensure the chain of finalization is

   * executed entirely.

   */

  function finalization() internal {

  }

}



contract RefundVault is Ownable {

  using SafeMath for uint256;



  enum State { Active, Refunding, Closed }



  mapping (address => uint256) public deposited;

  address public wallet;

  State public state;



  event Closed();

  event RefundsEnabled();

  event Refunded(address indexed beneficiary, uint256 weiAmount);



  function RefundVault(address _wallet) {

    require(_wallet != 0x0);

    wallet = _wallet;

    state = State.Active;

  }



  function deposit(address investor) onlyOwner payable {

    deposited[investor] = deposited[investor].add(msg.value);

  }



  function close() onlyOwner {

    require(state == State.Active);

    state = State.Closed;

    Closed();

    wallet.transfer(this.balance);

  }



  function enableRefunds() onlyOwner {

    require(state == State.Active);

    state = State.Refunding;

    RefundsEnabled();

  }



  function refund(address investor) {

    require(state == State.Refunding);

    uint256 depositedValue = deposited[investor];

    deposited[investor] = 0;

    investor.transfer(depositedValue);

    Refunded(investor, depositedValue);

  }

}



















contract Pausable is Ownable {

  event Pause();

  event Unpause();



  bool public paused = false;





  /**

   * @dev Modifier to make a function callable only when the contract is not paused.

   */

  modifier whenNotPaused() {

    require(!paused);

    _;

  }



  /**

   * @dev Modifier to make a function callable only when the contract is paused.

   */

  modifier whenPaused() {

    require(paused);

    _;

  }



  /**

   * @dev called by the owner to pause, triggers stopped state

   */

  function pause() onlyOwner whenNotPaused public {

    paused = true;

    Pause();

  }



  /**

   * @dev called by the owner to unpause, returns to normal state

   */

  function unpause() onlyOwner whenPaused public {

    paused = false;

    Unpause();

  }

}



contract PausableToken is StandardToken, Pausable {



  function transfer(address _to, uint256 _value) public whenNotPaused returns (bool) {

    return super.transfer(_to, _value);

  }



  function transferFrom(address _from, address _to, uint256 _value) public whenNotPaused returns (bool) {

    return super.transferFrom(_from, _to, _value);

  }



  function approve(address _spender, uint256 _value) public whenNotPaused returns (bool) {

    return super.approve(_spender, _value);

  }



  function increaseApproval(address _spender, uint _addedValue) public whenNotPaused returns (bool success) {

    return super.increaseApproval(_spender, _addedValue);

  }



  function decreaseApproval(address _spender, uint _subtractedValue) public whenNotPaused returns (bool success) {

    return super.decreaseApproval(_spender, _subtractedValue);

  }

}





















contract TokenRecipient {



    function tokenFallback(address sender, uint256 _value, bytes _extraData) returns (bool) {}



}











contract NPXSToken is MintableToken, BurnableToken, PausableToken {



    string public constant name = "Pundi X Token";

    string public constant symbol = "NPXS";

    uint8 public constant decimals = 18;





    function NPXSToken() {



    }





    // --------------------------------------------------------



    function transferFrom(address _from, address _to, uint256 _value) public whenNotPaused returns (bool) {

        bool result = super.transferFrom(_from, _to, _value);

        return result;

    }



    mapping (address => bool) stopReceive;



    function setStopReceive(bool stop) {

        stopReceive[msg.sender] = stop;

    }



    function getStopReceive() constant public returns (bool) {

        return stopReceive[msg.sender];

    }



    function transfer(address _to, uint256 _value) public whenNotPaused returns (bool) {

        require(!stopReceive[_to]);

        bool result = super.transfer(_to, _value);

        return result;

    }





    function mint(address _to, uint256 _amount) onlyOwner canMint public returns (bool) {

        bool result = super.mint(_to, _amount);

        return result;

    }



    function burn(uint256 _value) public {

        super.burn(_value);

    }



    // --------------------------------------------------------



    // 锁仓

    function pause() onlyOwner whenNotPaused public {

        super.pause();

    }



    // 解仓

    function unpause() onlyOwner whenPaused public {

        super.unpause();

    }



    function transferAndCall(address _recipient, uint256 _amount, bytes _data) {

        require(_recipient != address(0));

        require(_amount <= balances[msg.sender]);



        balances[msg.sender] = balances[msg.sender].sub(_amount);

        balances[_recipient] = balances[_recipient].add(_amount);



        require(TokenRecipient(_recipient).tokenFallback(msg.sender, _amount, _data));

        Transfer(msg.sender, _recipient, _amount);

    }



}