pragma solidity 0.4.20;



// No deps verison.



/**

 * @title ERC20

 * @dev A standard interface for tokens.

 * @dev https://github.com/ethereum/EIPs/blob/master/EIPS/eip-20-token-standard.md

 */

contract ERC20 {

  

    /// @dev Returns the total token supply

    function totalSupply() public constant returns (uint256 supply);



    /// @dev Returns the account balance of the account with address _owner

    function balanceOf(address _owner) public constant returns (uint256 balance);



    /// @dev Transfers _value number of tokens to address _to

    function transfer(address _to, uint256 _value) public returns (bool success);



    /// @dev Transfers _value number of tokens from address _from to address _to

    function transferFrom(address _from, address _to, uint256 _value) public returns (bool success);



    /// @dev Allows _spender to withdraw from the msg.sender's account up to the _value amount

    function approve(address _spender, uint256 _value) public returns (bool success);



    /// @dev Returns the amount which _spender is still allowed to withdraw from _owner

    function allowance(address _owner, address _spender) public constant returns (uint256 remaining);



    event Transfer(address indexed _from, address indexed _to, uint256 _value);

    event Approval(address indexed _owner, address indexed _spender, uint256 _value);



}



/// @title Owned

/// @author Adri Massanet <adria@codecontext.io>

/// @notice The Owned contract has an owner address, and provides basic 

///  authorization control functions, this simplifies & the implementation of

///  user permissions; this contract has three work flows for a change in

///  ownership, the first requires the new owner to validate that they have the

///  ability to accept ownership, the second allows the ownership to be

///  directly transfered without requiring acceptance, and the third allows for

///  the ownership to be removed to allow for decentralization 

contract Owned {



    address public owner;

    address public newOwnerCandidate;



    event OwnershipRequested(address indexed by, address indexed to);

    event OwnershipTransferred(address indexed from, address indexed to);

    event OwnershipRemoved();



    /// @dev The constructor sets the `msg.sender` as the`owner` of the contract

    function Owned() public {

        owner = msg.sender;

    }



    /// @dev `owner` is the only address that can call a function with this

    /// modifier

    modifier onlyOwner() {

        require (msg.sender == owner);

        _;

    }

    

    /// @dev In this 1st option for ownership transfer `proposeOwnership()` must

    ///  be called first by the current `owner` then `acceptOwnership()` must be

    ///  called by the `newOwnerCandidate`

    /// @notice `onlyOwner` Proposes to transfer control of the contract to a

    ///  new owner

    /// @param _newOwnerCandidate The address being proposed as the new owner

    function proposeOwnership(address _newOwnerCandidate) public onlyOwner {

        newOwnerCandidate = _newOwnerCandidate;

        OwnershipRequested(msg.sender, newOwnerCandidate);

    }



    /// @notice Can only be called by the `newOwnerCandidate`, accepts the

    ///  transfer of ownership

    function acceptOwnership() public {

        require(msg.sender == newOwnerCandidate);



        address oldOwner = owner;

        owner = newOwnerCandidate;

        newOwnerCandidate = 0x0;



        OwnershipTransferred(oldOwner, owner);

    }



    /// @dev In this 2nd option for ownership transfer `changeOwnership()` can

    ///  be called and it will immediately assign ownership to the `newOwner`

    /// @notice `owner` can step down and assign some other address to this role

    /// @param _newOwner The address of the new owner

    function changeOwnership(address _newOwner) public onlyOwner {

        require(_newOwner != 0x0);



        address oldOwner = owner;

        owner = _newOwner;

        newOwnerCandidate = 0x0;



        OwnershipTransferred(oldOwner, owner);

    }



    /// @dev In this 3rd option for ownership transfer `removeOwnership()` can

    ///  be called and it will immediately assign ownership to the 0x0 address;

    ///  it requires a 0xdece be input as a parameter to prevent accidental use

    /// @notice Decentralizes the contract, this operation cannot be undone 

    /// @param _dac `0xdac` has to be entered for this function to work

    function removeOwnership(address _dac) public onlyOwner {

        require(_dac == 0xdac);

        owner = 0x0;

        newOwnerCandidate = 0x0;

        OwnershipRemoved();     

    }

} 



/// @dev `Escapable` is a base level contract built off of the `Owned`

///  contract; it creates an escape hatch function that can be called in an

///  emergency that will allow designated addresses to send any ether or tokens

///  held in the contract to an `escapeHatchDestination` as long as they were

///  not blacklisted

contract Escapable is Owned {

    address public escapeHatchCaller;

    address public escapeHatchDestination;

    mapping (address=>bool) private escapeBlacklist; // Token contract addresses



    /// @notice The Constructor assigns the `escapeHatchDestination` and the

    ///  `escapeHatchCaller`

    /// @param _escapeHatchCaller The address of a trusted account or contract

    ///  to call `escapeHatch()` to send the ether in this contract to the

    ///  `escapeHatchDestination` it would be ideal that `escapeHatchCaller`

    ///  cannot move funds out of `escapeHatchDestination`

    /// @param _escapeHatchDestination The address of a safe location (usu a

    ///  Multisig) to send the ether held in this contract; if a neutral address

    ///  is required, the WHG Multisig is an option:

    ///  0x8Ff920020c8AD673661c8117f2855C384758C572 

    function Escapable(address _escapeHatchCaller, address _escapeHatchDestination) public {

        escapeHatchCaller = _escapeHatchCaller;

        escapeHatchDestination = _escapeHatchDestination;

    }



    /// @dev The addresses preassigned as `escapeHatchCaller` or `owner`

    ///  are the only addresses that can call a function with this modifier

    modifier onlyEscapeHatchCallerOrOwner {

        require ((msg.sender == escapeHatchCaller)||(msg.sender == owner));

        _;

    }



    /// @notice Creates the blacklist of tokens that are not able to be taken

    ///  out of the contract; can only be done at the deployment, and the logic

    ///  to add to the blacklist will be in the constructor of a child contract

    /// @param _token the token contract address that is to be blacklisted 

    function blacklistEscapeToken(address _token) internal {

        escapeBlacklist[_token] = true;

        EscapeHatchBlackistedToken(_token);

    }



    /// @notice Checks to see if `_token` is in the blacklist of tokens

    /// @param _token the token address being queried

    /// @return False if `_token` is in the blacklist and can't be taken out of

    ///  the contract via the `escapeHatch()`

    function isTokenEscapable(address _token) view public returns (bool) {

        return !escapeBlacklist[_token];

    }



    /// @notice The `escapeHatch()` should only be called as a last resort if a

    /// security issue is uncovered or something unexpected happened

    /// @param _token to transfer, use 0x0 for ether

    function escapeHatch(address _token) public onlyEscapeHatchCallerOrOwner {   

        require(escapeBlacklist[_token]==false);



        uint256 balance;



        /// @dev Logic for ether

        if (_token == 0x0) {

            balance = this.balance;

            escapeHatchDestination.transfer(balance);

            EscapeHatchCalled(_token, balance);

            return;

        }

        /// @dev Logic for tokens

        ERC20 token = ERC20(_token);

        balance = token.balanceOf(this);

        require(token.transfer(escapeHatchDestination, balance));

        EscapeHatchCalled(_token, balance);

    }



    /// @notice Changes the address assigned to call `escapeHatch()`

    /// @param _newEscapeHatchCaller The address of a trusted account or

    ///  contract to call `escapeHatch()` to send the value in this contract to

    ///  the `escapeHatchDestination`; it would be ideal that `escapeHatchCaller`

    ///  cannot move funds out of `escapeHatchDestination`

    function changeHatchEscapeCaller(address _newEscapeHatchCaller) public onlyEscapeHatchCallerOrOwner {

        escapeHatchCaller = _newEscapeHatchCaller;

    }



    event EscapeHatchBlackistedToken(address token);

    event EscapeHatchCalled(address token, uint amount);

}



// TightlyPacked is cheaper if you need to store input data and if amount is less than 12 bytes.

// Normal is cheaper if you don't need to store input data or if amounts are greater than 12 bytes.

contract UnsafeMultiplexor is Escapable(0, 0) {

    function init(address _escapeHatchCaller, address _escapeHatchDestination) public {

        require(escapeHatchCaller == 0);

        require(_escapeHatchCaller != 0);

        require(_escapeHatchDestination != 0);

        escapeHatchCaller = _escapeHatchCaller;

        escapeHatchDestination = _escapeHatchDestination;

    }

    

    modifier sendBackLeftEther() {

        uint balanceBefore = this.balance - msg.value;

        _;

        uint leftovers = this.balance - balanceBefore;

        if (leftovers > 0) {

            msg.sender.transfer(leftovers);

        }

    }

    

    function multiTransferTightlyPacked(bytes32[] _addressAndAmount) sendBackLeftEther() payable public returns(bool) {

        for (uint i = 0; i < _addressAndAmount.length; i++) {

            _unsafeTransfer(address(_addressAndAmount[i] >> 96), uint(uint96(_addressAndAmount[i])));

        }

        return true;

    }



    function multiTransfer(address[] _address, uint[] _amount) sendBackLeftEther() payable public returns(bool) {

        for (uint i = 0; i < _address.length; i++) {

            _unsafeTransfer(_address[i], _amount[i]);

        }

        return true;

    }



    function multiCallTightlyPacked(bytes32[] _addressAndAmount) sendBackLeftEther() payable public returns(bool) {

        for (uint i = 0; i < _addressAndAmount.length; i++) {

            _unsafeCall(address(_addressAndAmount[i] >> 96), uint(uint96(_addressAndAmount[i])));

        }

        return true;

    }



    function multiCall(address[] _address, uint[] _amount) sendBackLeftEther() payable public returns(bool) {

        for (uint i = 0; i < _address.length; i++) {

            _unsafeCall(_address[i], _amount[i]);

        }

        return true;

    }



    function _unsafeTransfer(address _to, uint _amount) internal {

        require(_to != 0);

        _to.send(_amount);

    }



    function _unsafeCall(address _to, uint _amount) internal {

        require(_to != 0);

        _to.call.value(_amount)();

    }

}