pragma solidity ^0.5.15;



/**

 * @dev Interface of the ERC20 standard as defined in the EIP.

 */

interface IERC20 {

    /**

     * @dev Returns the amount of tokens in existence.

     */

    function totalSupply() external view returns (uint256);



    /**

     * @dev Returns the amount of tokens owned by `account`.

     */

    function balanceOf(address account) external view returns (uint256);



    /**

     * @dev Moves `amount` tokens from the caller's account to `recipient`.

     *

     * Returns a boolean value indicating whether the operation succeeded.

     *

     * Emits a {Transfer} event.

     */

    function transfer(address recipient, uint256 amount) external returns (bool);



    /**

     * @dev Returns the remaining number of tokens that `spender` will be

     * allowed to spend on behalf of `owner` through {transferFrom}. This is

     * zero by default.

     *

     * This value changes when {approve} or {transferFrom} are called.

     */

    function allowance(address owner, address spender) external view returns (uint256);



    /**

     * @dev Sets `amount` as the allowance of `spender` over the caller's tokens.

     *

     * Returns a boolean value indicating whether the operation succeeded.

     *

     * IMPORTANT: Beware that changing an allowance with this method brings the risk

     * that someone may use both the old and the new allowance by unfortunate

     * transaction ordering. One possible solution to mitigate this race

     * condition is to first reduce the spender's allowance to 0 and set the

     * desired value afterwards:

     * https://github.com/ethereum/EIPs/issues/20#issuecomment-263524729

     *

     * Emits an {Approval} event.

     */

    function approve(address spender, uint256 amount) external returns (bool);



    /**

     * @dev Moves `amount` tokens from `sender` to `recipient` using the

     * allowance mechanism. `amount` is then deducted from the caller's

     * allowance.

     *

     * Returns a boolean value indicating whether the operation succeeded.

     *

     * Emits a {Transfer} event.

     */

    function transferFrom(address sender, address recipient, uint256 amount) external returns (bool);



    /**

     * @dev Emitted when `value` tokens are moved from one account (`from`) to

     * another (`to`).

     *

     * Note that `value` may be zero.

     */

    event Transfer(address indexed from, address indexed to, uint256 value);



    /**

     * @dev Emitted when the allowance of a `spender` for an `owner` is set by

     * a call to {approve}. `value` is the new allowance.

     */

    event Approval(address indexed owner, address indexed spender, uint256 value);

}



/**

 * @dev Wrappers over Solidity's arithmetic operations with added overflow

 * checks.

 *

 * Arithmetic operations in Solidity wrap on overflow. This can easily result

 * in bugs, because programmers usually assume that an overflow raises an

 * error, which is the standard behavior in high level programming languages.

 * `SafeMath` restores this intuition by reverting the transaction when an

 * operation overflows.

 *

 * Using this library instead of the unchecked operations eliminates an entire

 * class of bugs, so it's recommended to use it always.

 */

library SafeMath {

    /**

     * @dev Returns the addition of two unsigned integers, reverting on

     * overflow.

     *

     * Counterpart to Solidity's `+` operator.

     *

     * Requirements:

     *

     * - Addition cannot overflow.

     */

    function add(uint256 a, uint256 b) internal pure returns (uint256) {

        uint256 c = a + b;

        require(c >= a, "SafeMath: addition overflow");



        return c;

    }



    /**

     * @dev Returns the subtraction of two unsigned integers, reverting on

     * overflow (when the result is negative).

     *

     * Counterpart to Solidity's `-` operator.

     *

     * Requirements:

     *

     * - Subtraction cannot overflow.

     */

    function sub(uint256 a, uint256 b) internal pure returns (uint256) {

        return sub(a, b, "SafeMath: subtraction overflow");

    }



    /**

     * @dev Returns the subtraction of two unsigned integers, reverting with custom message on

     * overflow (when the result is negative).

     *

     * Counterpart to Solidity's `-` operator.

     *

     * Requirements:

     *

     * - Subtraction cannot overflow.

     */

    function sub(uint256 a, uint256 b, string memory errorMessage) internal pure returns (uint256) {

        require(b <= a, errorMessage);

        uint256 c = a - b;



        return c;

    }



    /**

     * @dev Returns the multiplication of two unsigned integers, reverting on

     * overflow.

     *

     * Counterpart to Solidity's `*` operator.

     *

     * Requirements:

     *

     * - Multiplication cannot overflow.

     */

    function mul(uint256 a, uint256 b) internal pure returns (uint256) {

        // Gas optimization: this is cheaper than requiring 'a' not being zero, but the

        // benefit is lost if 'b' is also tested.

        // See: https://github.com/OpenZeppelin/openzeppelin-contracts/pull/522

        if (a == 0) {

            return 0;

        }



        uint256 c = a * b;

        require(c / a == b, "SafeMath: multiplication overflow");



        return c;

    }



    /**

     * @dev Returns the integer division of two unsigned integers. Reverts on

     * division by zero. The result is rounded towards zero.

     *

     * Counterpart to Solidity's `/` operator. Note: this function uses a

     * `revert` opcode (which leaves remaining gas untouched) while Solidity

     * uses an invalid opcode to revert (consuming all remaining gas).

     *

     * Requirements:

     *

     * - The divisor cannot be zero.

     */

    function div(uint256 a, uint256 b) internal pure returns (uint256) {

        return div(a, b, "SafeMath: division by zero");

    }



    /**

     * @dev Returns the integer division of two unsigned integers. Reverts with custom message on

     * division by zero. The result is rounded towards zero.

     *

     * Counterpart to Solidity's `/` operator. Note: this function uses a

     * `revert` opcode (which leaves remaining gas untouched) while Solidity

     * uses an invalid opcode to revert (consuming all remaining gas).

     *

     * Requirements:

     *

     * - The divisor cannot be zero.

     */

    function div(uint256 a, uint256 b, string memory errorMessage) internal pure returns (uint256) {

        require(b > 0, errorMessage);

        uint256 c = a / b;

        // assert(a == b * c + a % b); // There is no case in which this doesn't hold



        return c;

    }



    /**

     * @dev Returns the remainder of dividing two unsigned integers. (unsigned integer modulo),

     * Reverts when dividing by zero.

     *

     * Counterpart to Solidity's `%` operator. This function uses a `revert`

     * opcode (which leaves remaining gas untouched) while Solidity uses an

     * invalid opcode to revert (consuming all remaining gas).

     *

     * Requirements:

     *

     * - The divisor cannot be zero.

     */

    function mod(uint256 a, uint256 b) internal pure returns (uint256) {

        return mod(a, b, "SafeMath: modulo by zero");

    }



    /**

     * @dev Returns the remainder of dividing two unsigned integers. (unsigned integer modulo),

     * Reverts with custom message when dividing by zero.

     *

     * Counterpart to Solidity's `%` operator. This function uses a `revert`

     * opcode (which leaves remaining gas untouched) while Solidity uses an

     * invalid opcode to revert (consuming all remaining gas).

     *

     * Requirements:

     *

     * - The divisor cannot be zero.

     */

    function mod(uint256 a, uint256 b, string memory errorMessage) internal pure returns (uint256) {

        require(b != 0, errorMessage);

        return a % b;

    }

}



/**

 * @dev Collection of functions related to the address type

 */

library Address {

    /**

     * @dev Returns true if `account` is a contract.

     *

     * [IMPORTANT]

     * ====

     * It is unsafe to assume that an address for which this function returns

     * false is an externally-owned account (EOA) and not a contract.

     *

     * Among others, `isContract` will return false for the following

     * types of addresses:

     *

     *  - an externally-owned account

     *  - a contract in construction

     *  - an address where a contract will be created

     *  - an address where a contract lived, but was destroyed

     * ====

     */

    function isContract(address account) internal view returns (bool) {

        // This method relies in extcodesize, which returns 0 for contracts in

        // construction, since the code is only stored at the end of the

        // constructor execution.



        uint256 size;

        // solhint-disable-next-line no-inline-assembly

        assembly { size := extcodesize(account) }

        return size > 0;

    }



    /**

     * @dev Replacement for Solidity's `transfer`: sends `amount` wei to

     * `recipient`, forwarding all available gas and reverting on errors.

     *

     * https://eips.ethereum.org/EIPS/eip-1884[EIP1884] increases the gas cost

     * of certain opcodes, possibly making contracts go over the 2300 gas limit

     * imposed by `transfer`, making them unable to receive funds via

     * `transfer`. {sendValue} removes this limitation.

     *

     * https://diligence.consensys.net/posts/2019/09/stop-using-soliditys-transfer-now/[Learn more].

     *

     * IMPORTANT: because control is transferred to `recipient`, care must be

     * taken to not create reentrancy vulnerabilities. Consider using

     * {ReentrancyGuard} or the

     * https://solidity.readthedocs.io/en/v0.5.11/security-considerations.html#use-the-checks-effects-interactions-pattern[checks-effects-interactions pattern].

     */

    function sendValue(address payable recipient, uint256 amount) internal {

        require(address(this).balance >= amount, "Address: insufficient balance");



        // solhint-disable-next-line avoid-low-level-calls, avoid-call-value

        (bool success, ) = recipient.call.value(amount)("");

        require(success, "Address: unable to send value, recipient may have reverted");

    }



    /**

     * @dev Performs a Solidity function call using a low level `call`. A

     * plain`call` is an unsafe replacement for a function call: use this

     * function instead.

     *

     * If `target` reverts with a revert reason, it is bubbled up by this

     * function (like regular Solidity function calls).

     *

     * Returns the raw returned data. To convert to the expected return value,

     * use https://solidity.readthedocs.io/en/latest/units-and-global-variables.html?highlight=abi.decode#abi-encoding-and-decoding-functions[`abi.decode`].

     *

     * Requirements:

     *

     * - `target` must be a contract.

     * - calling `target` with `data` must not revert.

     *

     * _Available since v3.1._

     */

    function functionCall(address target, bytes memory data) internal returns (bytes memory) {

      return functionCall(target, data, "Address: low-level call failed");

    }



    /**

     * @dev Same as {xref-Address-functionCall-address-bytes-}[`functionCall`], but with

     * `errorMessage` as a fallback revert reason when `target` reverts.

     *

     * _Available since v3.1._

     */

    function functionCall(address target, bytes memory data, string memory errorMessage) internal returns (bytes memory) {

        return _functionCallWithValue(target, data, 0, errorMessage);

    }



    /**

     * @dev Same as {xref-Address-functionCall-address-bytes-}[`functionCall`],

     * but also transferring `value` wei to `target`.

     *

     * Requirements:

     *

     * - the calling contract must have an ETH balance of at least `value`.

     * - the called Solidity function must be `payable`.

     *

     * _Available since v3.1._

     */

    function functionCallWithValue(address target, bytes memory data, uint256 value) internal returns (bytes memory) {

        return functionCallWithValue(target, data, value, "Address: low-level call with value failed");

    }



    /**

     * @dev Same as {xref-Address-functionCallWithValue-address-bytes-uint256-}[`functionCallWithValue`], but

     * with `errorMessage` as a fallback revert reason when `target` reverts.

     *

     * _Available since v3.1._

     */

    function functionCallWithValue(address target, bytes memory data, uint256 value, string memory errorMessage) internal returns (bytes memory) {

        require(address(this).balance >= value, "Address: insufficient balance for call");

        return _functionCallWithValue(target, data, value, errorMessage);

    }



    function _functionCallWithValue(address target, bytes memory data, uint256 weiValue, string memory errorMessage) private returns (bytes memory) {

        require(isContract(target), "Address: call to non-contract");



        // solhint-disable-next-line avoid-low-level-calls

        (bool success, bytes memory returndata) = target.call.value(weiValue)(data);

        if (success) {

            return returndata;

        } else {

            // Look for revert reason and bubble it up if present

            if (returndata.length > 0) {

                // The easiest way to bubble the revert reason is using memory via assembly



                // solhint-disable-next-line no-inline-assembly

                assembly {

                    let returndata_size := mload(returndata)

                    revert(add(32, returndata), returndata_size)

                }

            } else {

                revert(errorMessage);

            }

        }

    }

}



/**

 * @title SafeERC20

 * @dev Wrappers around ERC20 operations that throw on failure (when the token

 * contract returns false). Tokens that return no value (and instead revert or

 * throw on failure) are also supported, non-reverting calls are assumed to be

 * successful.

 * To use this library you can add a `using SafeERC20 for IERC20;` statement to your contract,

 * which allows you to call the safe operations as `token.safeTransfer(...)`, etc.

 */

library SafeERC20 {

    using SafeMath for uint256;

    using Address for address;



    function safeTransfer(IERC20 token, address to, uint256 value) internal {

        _callOptionalReturn(token, abi.encodeWithSelector(token.transfer.selector, to, value));

    }



    function safeTransferFrom(IERC20 token, address from, address to, uint256 value) internal {

        _callOptionalReturn(token, abi.encodeWithSelector(token.transferFrom.selector, from, to, value));

    }



    /**

     * @dev Deprecated. This function has issues similar to the ones found in

     * {IERC20-approve}, and its usage is discouraged.

     *

     * Whenever possible, use {safeIncreaseAllowance} and

     * {safeDecreaseAllowance} instead.

     */

    function safeApprove(IERC20 token, address spender, uint256 value) internal {

        // safeApprove should only be called when setting an initial allowance,

        // or when resetting it to zero. To increase and decrease it, use

        // 'safeIncreaseAllowance' and 'safeDecreaseAllowance'

        // solhint-disable-next-line max-line-length

        require((value == 0) || (token.allowance(address(this), spender) == 0),

            "SafeERC20: approve from non-zero to non-zero allowance"

        );

        _callOptionalReturn(token, abi.encodeWithSelector(token.approve.selector, spender, value));

    }



    function safeIncreaseAllowance(IERC20 token, address spender, uint256 value) internal {

        uint256 newAllowance = token.allowance(address(this), spender).add(value);

        _callOptionalReturn(token, abi.encodeWithSelector(token.approve.selector, spender, newAllowance));

    }



    function safeDecreaseAllowance(IERC20 token, address spender, uint256 value) internal {

        uint256 newAllowance = token.allowance(address(this), spender).sub(value, "SafeERC20: decreased allowance below zero");

        _callOptionalReturn(token, abi.encodeWithSelector(token.approve.selector, spender, newAllowance));

    }



    /**

     * @dev Imitates a Solidity high-level call (i.e. a regular function call to a contract), relaxing the requirement

     * on the return value: the return value is optional (but if data is returned, it must not be false).

     * @param token The token targeted by the call.

     * @param data The call data (encoded using abi.encode or one of its variants).

     */

    function _callOptionalReturn(IERC20 token, bytes memory data) private {

        // We need to perform a low level call here, to bypass Solidity's return data size checking mechanism, since

        // we're implementing it ourselves. We use {Address.functionCall} to perform this call, which verifies that

        // the target address contains contract code and also asserts for success in the low-level call.



        bytes memory returndata = address(token).functionCall(data, "SafeERC20: low-level call failed");

        if (returndata.length > 0) { // Return data is optional

            // solhint-disable-next-line max-line-length

            require(abi.decode(returndata, (bool)), "SafeERC20: ERC20 operation did not succeed");

        }

    }

}



/* Copyright 2020 Compound Labs, Inc.



Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:



1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.



2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.



3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.



THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */





contract YAMGovernanceStorage {

    /// @notice A record of each accounts delegate

    mapping (address => address) internal _delegates;



    /// @notice A checkpoint for marking number of votes from a given block

    struct Checkpoint {

        uint32 fromBlock;

        uint256 votes;

    }



    /// @notice A record of votes checkpoints for each account, by index

    mapping (address => mapping (uint32 => Checkpoint)) public checkpoints;



    /// @notice The number of checkpoints for each account

    mapping (address => uint32) public numCheckpoints;



    /// @notice The EIP-712 typehash for the contract's domain

    bytes32 public constant DOMAIN_TYPEHASH = keccak256("EIP712Domain(string name,uint256 chainId,address verifyingContract)");



    /// @notice The EIP-712 typehash for the delegation struct used by the contract

    bytes32 public constant DELEGATION_TYPEHASH = keccak256("Delegation(address delegatee,uint256 nonce,uint256 expiry)");



    /// @notice A record of states for signing / validating signatures

    mapping (address => uint) public nonces;

}



// Storage for a YAM token

contract YAMTokenStorage {



    using SafeMath for uint256;



    /**

     * @dev Guard variable for re-entrancy checks. Not currently used

     */

    bool internal _notEntered;



    /**

     * @notice EIP-20 token name for this token

     */

    string public name;



    /**

     * @notice EIP-20 token symbol for this token

     */

    string public symbol;



    /**

     * @notice EIP-20 token decimals for this token

     */

    uint8 public decimals;



    /**

     * @notice Governor for this contract

     */

    address public gov;



    /**

     * @notice Pending governance for this contract

     */

    address public pendingGov;



    /**

     * @notice Approved rebaser for this contract

     */

    address public rebaser;



    /**

     * @notice Approved migrator for this contract

     */

    address public migrator;



    /**

     * @notice Incentivizer address of YAM protocol

     */

    address public incentivizer;



    /**

     * @notice Total supply of YAMs

     */

    uint256 public totalSupply;



    /**

     * @notice Internal decimals used to handle scaling factor

     */

    uint256 public constant internalDecimals = 10**24;



    /**

     * @notice Used for percentage maths

     */

    uint256 public constant BASE = 10**18;



    /**

     * @notice Scaling factor that adjusts everyone's balances

     */

    uint256 public yamsScalingFactor;



    mapping (address => uint256) internal _yamBalances;



    mapping (address => mapping (address => uint256)) internal _allowedFragments;



    uint256 public initSupply;





    // keccak256("Permit(address owner,address spender,uint256 value,uint256 nonce,uint256 deadline)");

    bytes32 public constant PERMIT_TYPEHASH = 0x6e71edae12b1b97f4d1f60370fef10105fa2faae0126114a169c64845d6126c9;

    bytes32 public DOMAIN_SEPARATOR;

}



contract YAMTokenInterface is YAMTokenStorage, YAMGovernanceStorage {



    /// @notice An event thats emitted when an account changes its delegate

    event DelegateChanged(address indexed delegator, address indexed fromDelegate, address indexed toDelegate);



    /// @notice An event thats emitted when a delegate account's vote balance changes

    event DelegateVotesChanged(address indexed delegate, uint previousBalance, uint newBalance);



    /**

     * @notice Event emitted when tokens are rebased

     */

    event Rebase(uint256 epoch, uint256 prevYamsScalingFactor, uint256 newYamsScalingFactor);



    /*** Gov Events ***/



    /**

     * @notice Event emitted when pendingGov is changed

     */

    event NewPendingGov(address oldPendingGov, address newPendingGov);



    /**

     * @notice Event emitted when gov is changed

     */

    event NewGov(address oldGov, address newGov);



    /**

     * @notice Sets the rebaser contract

     */

    event NewRebaser(address oldRebaser, address newRebaser);



    /**

     * @notice Sets the migrator contract

     */

    event NewMigrator(address oldMigrator, address newMigrator);



    /**

     * @notice Sets the incentivizer contract

     */

    event NewIncentivizer(address oldIncentivizer, address newIncentivizer);



    /* - ERC20 Events - */



    /**

     * @notice EIP20 Transfer event

     */

    event Transfer(address indexed from, address indexed to, uint amount);



    /**

     * @notice EIP20 Approval event

     */

    event Approval(address indexed owner, address indexed spender, uint amount);



    /* - Extra Events - */

    /**

     * @notice Tokens minted event

     */

    event Mint(address to, uint256 amount);



    // Public functions

    function transfer(address to, uint256 value) external returns(bool);

    function transferFrom(address from, address to, uint256 value) external returns(bool);

    function balanceOf(address who) external view returns(uint256);

    function balanceOfUnderlying(address who) external view returns(uint256);

    function allowance(address owner_, address spender) external view returns(uint256);

    function approve(address spender, uint256 value) external returns (bool);

    function increaseAllowance(address spender, uint256 addedValue) external returns (bool);

    function decreaseAllowance(address spender, uint256 subtractedValue) external returns (bool);

    function maxScalingFactor() external view returns (uint256);

    function yamToFragment(uint256 yam) external view returns (uint256);

    function fragmentToYam(uint256 value) external view returns (uint256);



    /* - Governance Functions - */

    function getPriorVotes(address account, uint blockNumber) external view returns (uint256);

    function delegateBySig(address delegatee, uint nonce, uint expiry, uint8 v, bytes32 r, bytes32 s) external;

    function delegate(address delegatee) external;

    function delegates(address delegator) external view returns (address);

    function getCurrentVotes(address account) external view returns (uint256);



    /* - Permissioned/Governance functions - */

    function mint(address to, uint256 amount) external returns (bool);

    function rebase(uint256 epoch, uint256 indexDelta, bool positive) external returns (uint256);

    function _setRebaser(address rebaser_) external;

    function _setIncentivizer(address incentivizer_) external;

    function _setPendingGov(address pendingGov_) external;

    function _acceptGov() external;

}



/* Copyright 2020 Compound Labs, Inc.



Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:



1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.



2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.



3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.



THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */







contract YAMGovernanceToken is YAMTokenInterface {



      /// @notice An event thats emitted when an account changes its delegate

    event DelegateChanged(address indexed delegator, address indexed fromDelegate, address indexed toDelegate);



    /// @notice An event thats emitted when a delegate account's vote balance changes

    event DelegateVotesChanged(address indexed delegate, uint previousBalance, uint newBalance);



    /**

     * @notice Get delegatee for an address delegating

     * @param delegator The address to get delegatee for

     */

    function delegates(address delegator)

        external

        view

        returns (address)

    {

        return _delegates[delegator];

    }



   /**

    * @notice Delegate votes from `msg.sender` to `delegatee`

    * @param delegatee The address to delegate votes to

    */

    function delegate(address delegatee) external {

        return _delegate(msg.sender, delegatee);

    }



    /**

     * @notice Delegates votes from signatory to `delegatee`

     * @param delegatee The address to delegate votes to

     * @param nonce The contract state required to match the signature

     * @param expiry The time at which to expire the signature

     * @param v The recovery byte of the signature

     * @param r Half of the ECDSA signature pair

     * @param s Half of the ECDSA signature pair

     */

    function delegateBySig(

        address delegatee,

        uint nonce,

        uint expiry,

        uint8 v,

        bytes32 r,

        bytes32 s

    )

        external

    {

        bytes32 structHash = keccak256(

            abi.encode(

                DELEGATION_TYPEHASH,

                delegatee,

                nonce,

                expiry

            )

        );



        bytes32 digest = keccak256(

            abi.encodePacked(

                "\x19\x01",

                DOMAIN_SEPARATOR,

                structHash

            )

        );



        address signatory = ecrecover(digest, v, r, s);

        require(signatory != address(0), "YAM::delegateBySig: invalid signature");

        require(nonce == nonces[signatory]++, "YAM::delegateBySig: invalid nonce");

        require(now <= expiry, "YAM::delegateBySig: signature expired");

        return _delegate(signatory, delegatee);

    }



    /**

     * @notice Gets the current votes balance for `account`

     * @param account The address to get votes balance

     * @return The number of current votes for `account`

     */

    function getCurrentVotes(address account)

        external

        view

        returns (uint256)

    {

        uint32 nCheckpoints = numCheckpoints[account];

        return nCheckpoints > 0 ? checkpoints[account][nCheckpoints - 1].votes : 0;

    }



    /**

     * @notice Determine the prior number of votes for an account as of a block number

     * @dev Block number must be a finalized block or else this function will revert to prevent misinformation.

     * @param account The address of the account to check

     * @param blockNumber The block number to get the vote balance at

     * @return The number of votes the account had as of the given block

     */

    function getPriorVotes(address account, uint blockNumber)

        external

        view

        returns (uint256)

    {

        require(blockNumber < block.number, "YAM::getPriorVotes: not yet determined");



        uint32 nCheckpoints = numCheckpoints[account];

        if (nCheckpoints == 0) {

            return 0;

        }



        // First check most recent balance

        if (checkpoints[account][nCheckpoints - 1].fromBlock <= blockNumber) {

            return checkpoints[account][nCheckpoints - 1].votes;

        }



        // Next check implicit zero balance

        if (checkpoints[account][0].fromBlock > blockNumber) {

            return 0;

        }



        uint32 lower = 0;

        uint32 upper = nCheckpoints - 1;

        while (upper > lower) {

            uint32 center = upper - (upper - lower) / 2; // ceil, avoiding overflow

            Checkpoint memory cp = checkpoints[account][center];

            if (cp.fromBlock == blockNumber) {

                return cp.votes;

            } else if (cp.fromBlock < blockNumber) {

                lower = center;

            } else {

                upper = center - 1;

            }

        }

        return checkpoints[account][lower].votes;

    }



    function _delegate(address delegator, address delegatee)

        internal

    {

        address currentDelegate = _delegates[delegator];

        uint256 delegatorBalance = _yamBalances[delegator]; // balance of underlying YAMs (not scaled);

        _delegates[delegator] = delegatee;



        emit DelegateChanged(delegator, currentDelegate, delegatee);



        _moveDelegates(currentDelegate, delegatee, delegatorBalance);

    }



    function _moveDelegates(address srcRep, address dstRep, uint256 amount) internal {

        if (srcRep != dstRep && amount > 0) {

            if (srcRep != address(0)) {

                // decrease old representative

                uint32 srcRepNum = numCheckpoints[srcRep];

                uint256 srcRepOld = srcRepNum > 0 ? checkpoints[srcRep][srcRepNum - 1].votes : 0;

                uint256 srcRepNew = srcRepOld.sub(amount);

                _writeCheckpoint(srcRep, srcRepNum, srcRepOld, srcRepNew);

            }



            if (dstRep != address(0)) {

                // increase new representative

                uint32 dstRepNum = numCheckpoints[dstRep];

                uint256 dstRepOld = dstRepNum > 0 ? checkpoints[dstRep][dstRepNum - 1].votes : 0;

                uint256 dstRepNew = dstRepOld.add(amount);

                _writeCheckpoint(dstRep, dstRepNum, dstRepOld, dstRepNew);

            }

        }

    }



    function _writeCheckpoint(

        address delegatee,

        uint32 nCheckpoints,

        uint256 oldVotes,

        uint256 newVotes

    )

        internal

    {

        uint32 blockNumber = safe32(block.number, "YAM::_writeCheckpoint: block number exceeds 32 bits");



        if (nCheckpoints > 0 && checkpoints[delegatee][nCheckpoints - 1].fromBlock == blockNumber) {

            checkpoints[delegatee][nCheckpoints - 1].votes = newVotes;

        } else {

            checkpoints[delegatee][nCheckpoints] = Checkpoint(blockNumber, newVotes);

            numCheckpoints[delegatee] = nCheckpoints + 1;

        }



        emit DelegateVotesChanged(delegatee, oldVotes, newVotes);

    }



    function safe32(uint n, string memory errorMessage) internal pure returns (uint32) {

        require(n < 2**32, errorMessage);

        return uint32(n);

    }



    function getChainId() internal pure returns (uint) {

        uint256 chainId;

        assembly { chainId := chainid() }

        return chainId;

    }

}



contract YAMToken is YAMGovernanceToken {

    // Modifiers

    modifier onlyGov() {

        require(msg.sender == gov);

        _;

    }



    modifier onlyRebaser() {

        require(msg.sender == rebaser);

        _;

    }



    modifier onlyMinter() {

        require(

            msg.sender == rebaser

            || msg.sender == gov

            || msg.sender == incentivizer

            || msg.sender == migrator,

            "not minter"

        );

        _;

    }



    modifier validRecipient(address to) {

        require(to != address(0x0));

        require(to != address(this));

        _;

    }



    function initialize(

        string memory name_,

        string memory symbol_,

        uint8 decimals_

    )

        public

    {

        require(yamsScalingFactor == 0, "already initialized");

        name = name_;

        symbol = symbol_;

        decimals = decimals_;

    }





    /**

    * @notice Computes the current max scaling factor

    */

    function maxScalingFactor()

        external

        view

        returns (uint256)

    {

        return _maxScalingFactor();

    }



    function _maxScalingFactor()

        internal

        view

        returns (uint256)

    {

        // scaling factor can only go up to 2**256-1 = initSupply * yamsScalingFactor

        // this is used to check if yamsScalingFactor will be too high to compute balances when rebasing.

        return uint256(-1) / initSupply;

    }



    /**

    * @notice Mints new tokens, increasing totalSupply, initSupply, and a users balance.

    * @dev Limited to onlyMinter modifier

    */

    function mint(address to, uint256 amount)

        external

        returns (bool)

    {

        _mint(to, amount);

        return true;

    }



    function _mint(address to, uint256 amount)

        internal

    {

        // increase totalSupply

        totalSupply = totalSupply.add(amount);



        // get underlying value

        uint256 yamValue = _fragmentToYam(amount);



        // increase initSupply

        initSupply = initSupply.add(yamValue);



        // make sure the mint didnt push maxScalingFactor too low

        require(yamsScalingFactor <= _maxScalingFactor(), "max scaling factor too low");



        // add balance

        _yamBalances[to] = _yamBalances[to].add(yamValue);



        // add delegates to the minter

        _moveDelegates(address(0), _delegates[to], yamValue);

        emit Mint(to, amount);

        emit Transfer(address(0), to, amount);

    }



    /**

    * @notice Mints new tokens using underlying amount, increasing totalSupply, initSupply, and a users balance.

    * @dev Limited to onlyMinter modifier

    */

    function mintUnderlying(address to, uint256 amount)

        external

        returns (bool)

    {

        _mintUnderlying(to, amount);

        return true;

    }



    function _mintUnderlying(address to, uint256 amount)

        internal

    {



        // increase initSupply

        initSupply = initSupply.add(amount);



        // get external value

        uint256 scaledAmount = _yamToFragment(amount);



        // increase totalSupply

        totalSupply = totalSupply.add(scaledAmount);



        // make sure the mint didnt push maxScalingFactor too low

        require(yamsScalingFactor <= _maxScalingFactor(), "max scaling factor too low");



        // add balance

        _yamBalances[to] = _yamBalances[to].add(amount);



        // add delegates to the minter

        _moveDelegates(address(0), _delegates[to], amount);

        emit Mint(to, scaledAmount);

        emit Transfer(address(0), to, scaledAmount);

   

    }



    /**

     * @dev Transfer underlying balance to a specified address.

     * @param to The address to transfer to.

     * @param value The amount to be transferred.

     * @return True on success, false otherwise.

     */

    function transferUnderlying(address to, uint256 value)

        external

        validRecipient(to)

        returns (bool)

    {

        // sub from balance of sender

        _yamBalances[msg.sender] = _yamBalances[msg.sender].sub(value);



        // add to balance of receiver

        _yamBalances[to] = _yamBalances[to].add(value);

        emit Transfer(msg.sender, to, _yamToFragment(value));



        _moveDelegates(_delegates[msg.sender], _delegates[to], value);

        return true;

    }

    

    /* - ERC20 functionality - */



    /**

    * @dev Transfer tokens to a specified address.

    * @param to The address to transfer to.

    * @param value The amount to be transferred.

    * @return True on success, false otherwise.

    */

    function transfer(address to, uint256 value)

        external

        validRecipient(to)

        returns (bool)

    {

        // underlying balance is stored in yams, so divide by current scaling factor



        // note, this means as scaling factor grows, dust will be untransferrable.

        // minimum transfer value == yamsScalingFactor / 1e24;



        // get amount in underlying

        uint256 yamValue = _fragmentToYam(value);



        // sub from balance of sender

        _yamBalances[msg.sender] = _yamBalances[msg.sender].sub(yamValue);



        // add to balance of receiver

        _yamBalances[to] = _yamBalances[to].add(yamValue);

        emit Transfer(msg.sender, to, value);



        _moveDelegates(_delegates[msg.sender], _delegates[to], yamValue);

        return true;

    }



    /**

    * @dev Transfer tokens from one address to another.

    * @param from The address you want to send tokens from.

    * @param to The address you want to transfer to.

    * @param value The amount of tokens to be transferred.

    */

    function transferFrom(address from, address to, uint256 value)

        external

        validRecipient(to)

        returns (bool)

    {

        // decrease allowance

        _allowedFragments[from][msg.sender] = _allowedFragments[from][msg.sender].sub(value);



        // get value in yams

        uint256 yamValue = _fragmentToYam(value);



        // sub from from

        _yamBalances[from] = _yamBalances[from].sub(yamValue);

        _yamBalances[to] = _yamBalances[to].add(yamValue);

        emit Transfer(from, to, value);



        _moveDelegates(_delegates[from], _delegates[to], yamValue);

        return true;

    }



    /**

    * @param who The address to query.

    * @return The balance of the specified address.

    */

    function balanceOf(address who)

      external

      view

      returns (uint256)

    {

      return _yamToFragment(_yamBalances[who]);

    }



    /** @notice Currently returns the internal storage amount

    * @param who The address to query.

    * @return The underlying balance of the specified address.

    */

    function balanceOfUnderlying(address who)

      external

      view

      returns (uint256)

    {

      return _yamBalances[who];

    }



    /**

     * @dev Function to check the amount of tokens that an owner has allowed to a spender.

     * @param owner_ The address which owns the funds.

     * @param spender The address which will spend the funds.

     * @return The number of tokens still available for the spender.

     */

    function allowance(address owner_, address spender)

        external

        view

        returns (uint256)

    {

        return _allowedFragments[owner_][spender];

    }



    /**

     * @dev Approve the passed address to spend the specified amount of tokens on behalf of

     * msg.sender. This method is included for ERC20 compatibility.

     * increaseAllowance and decreaseAllowance should be used instead.

     * Changing an allowance with this method brings the risk that someone may transfer both

     * the old and the new allowance - if they are both greater than zero - if a transfer

     * transaction is mined before the later approve() call is mined.

     *

     * @param spender The address which will spend the funds.

     * @param value The amount of tokens to be spent.

     */

    function approve(address spender, uint256 value)

        external

        returns (bool)

    {

        _allowedFragments[msg.sender][spender] = value;

        emit Approval(msg.sender, spender, value);

        return true;

    }



    /**

     * @dev Increase the amount of tokens that an owner has allowed to a spender.

     * This method should be used instead of approve() to avoid the double approval vulnerability

     * described above.

     * @param spender The address which will spend the funds.

     * @param addedValue The amount of tokens to increase the allowance by.

     */

    function increaseAllowance(address spender, uint256 addedValue)

        external

        returns (bool)

    {

        _allowedFragments[msg.sender][spender] =

            _allowedFragments[msg.sender][spender].add(addedValue);

        emit Approval(msg.sender, spender, _allowedFragments[msg.sender][spender]);

        return true;

    }



    /**

     * @dev Decrease the amount of tokens that an owner has allowed to a spender.

     *

     * @param spender The address which will spend the funds.

     * @param subtractedValue The amount of tokens to decrease the allowance by.

     */

    function decreaseAllowance(address spender, uint256 subtractedValue)

        external

        returns (bool)

    {

        uint256 oldValue = _allowedFragments[msg.sender][spender];

        if (subtractedValue >= oldValue) {

            _allowedFragments[msg.sender][spender] = 0;

        } else {

            _allowedFragments[msg.sender][spender] = oldValue.sub(subtractedValue);

        }

        emit Approval(msg.sender, spender, _allowedFragments[msg.sender][spender]);

        return true;

    }





    // --- Approve by signature ---

    function permit(

        address owner,

        address spender,

        uint256 value,

        uint256 deadline,

        uint8 v,

        bytes32 r,

        bytes32 s

    )

        external

    {

        require(now <= deadline, "YAM/permit-expired");



        bytes32 digest =

            keccak256(

                abi.encodePacked(

                    "\x19\x01",

                    DOMAIN_SEPARATOR,

                    keccak256(

                        abi.encode(

                            PERMIT_TYPEHASH,

                            owner,

                            spender,

                            value,

                            nonces[owner]++,

                            deadline

                        )

                    )

                )

            );



        require(owner != address(0), "YAM/invalid-address-0");

        require(owner == ecrecover(digest, v, r, s), "YAM/invalid-permit");

        _allowedFragments[owner][spender] = value;

        emit Approval(owner, spender, value);

    }



    /* - Governance Functions - */



    /** @notice sets the rebaser

     * @param rebaser_ The address of the rebaser contract to use for authentication.

     */

    function _setRebaser(address rebaser_)

        external

        onlyGov

    {

        address oldRebaser = rebaser;

        rebaser = rebaser_;

        emit NewRebaser(oldRebaser, rebaser_);

    }



    /** @notice sets the migrator

     * @param migrator_ The address of the migrator contract to use for authentication.

     */

    function _setMigrator(address migrator_)

        external

        onlyGov

    {

        address oldMigrator = migrator_;

        migrator = migrator_;

        emit NewMigrator(oldMigrator, migrator_);

    }



    /** @notice sets the incentivizer

     * @param incentivizer_ The address of the rebaser contract to use for authentication.

     */

    function _setIncentivizer(address incentivizer_)

        external

        onlyGov

    {

        address oldIncentivizer = incentivizer;

        incentivizer = incentivizer_;

        emit NewIncentivizer(oldIncentivizer, incentivizer_);

    }



    /** @notice sets the pendingGov

     * @param pendingGov_ The address of the rebaser contract to use for authentication.

     */

    function _setPendingGov(address pendingGov_)

        external

        onlyGov

    {

        address oldPendingGov = pendingGov;

        pendingGov = pendingGov_;

        emit NewPendingGov(oldPendingGov, pendingGov_);

    }



    /** @notice allows governance to assign delegate to self

     *

     */

    function _acceptGov()

        external

    {

        require(msg.sender == pendingGov, "!pending");

        address oldGov = gov;

        gov = pendingGov;

        pendingGov = address(0);

        emit NewGov(oldGov, gov);

    }



    /* - Extras - */



    /**

    * @notice Initiates a new rebase operation, provided the minimum time period has elapsed.

    *

    * @dev The supply adjustment equals (totalSupply * DeviationFromTargetRate) / rebaseLag

    *      Where DeviationFromTargetRate is (MarketOracleRate - targetRate) / targetRate

    *      and targetRate is CpiOracleRate / baseCpi

    */

    function rebase(

        uint256 epoch,

        uint256 indexDelta,

        bool positive

    )

        external

        onlyRebaser

        returns (uint256)

    {

        // no change

        if (indexDelta == 0) {

          emit Rebase(epoch, yamsScalingFactor, yamsScalingFactor);

          return totalSupply;

        }



        // for events

        uint256 prevYamsScalingFactor = yamsScalingFactor;





        if (!positive) {

            // negative rebase, decrease scaling factor

            yamsScalingFactor = yamsScalingFactor.mul(BASE.sub(indexDelta)).div(BASE);

        } else {

            // positive reabse, increase scaling factor

            uint256 newScalingFactor = yamsScalingFactor.mul(BASE.add(indexDelta)).div(BASE);

            if (newScalingFactor < _maxScalingFactor()) {

                yamsScalingFactor = newScalingFactor;

            } else {

                yamsScalingFactor = _maxScalingFactor();

            }

        }



        // update total supply, correctly

        totalSupply = _yamToFragment(initSupply);



        emit Rebase(epoch, prevYamsScalingFactor, yamsScalingFactor);

        return totalSupply;

    }



    function yamToFragment(uint256 yam)

        external

        view

        returns (uint256)

    {

        return _yamToFragment(yam);

    }



    function fragmentToYam(uint256 value)

        external

        view

        returns (uint256)

    {

        return _fragmentToYam(value);

    }



    function _yamToFragment(uint256 yam)

        internal

        view

        returns (uint256)

    {

        return yam.mul(yamsScalingFactor).div(internalDecimals);

    }



    function _fragmentToYam(uint256 value)

        internal

        view

        returns (uint256)

    {

        return value.mul(internalDecimals).div(yamsScalingFactor);

    }



    // Rescue tokens

    function rescueTokens(

        address token,

        address to,

        uint256 amount

    )

        external

        onlyGov

        returns (bool)

    {

        // transfer to

        SafeERC20.safeTransfer(IERC20(token), to, amount);

        return true;

    }

}



contract YAMLogic3 is YAMToken {

    /**

     * @notice Initialize the new money market

     * @param name_ ERC-20 name of this token

     * @param symbol_ ERC-20 symbol of this token

     * @param decimals_ ERC-20 decimal precision of this token

     */

    function initialize(

        string memory name_,

        string memory symbol_,

        uint8 decimals_,

        address initial_owner,

        uint256 initTotalSupply_

    )

        public

    {

        super.initialize(name_, symbol_, decimals_);



        yamsScalingFactor = 2501323450897068507;

        initSupply = _fragmentToYam(initTotalSupply_);

        totalSupply = initTotalSupply_;

        _yamBalances[initial_owner] = initSupply;



        DOMAIN_SEPARATOR = keccak256(

            abi.encode(

                DOMAIN_TYPEHASH,

                keccak256(bytes(name)),

                getChainId(),

                address(this)

            )

        );

    }

}



/* Copyright 2020 Compound Labs, Inc.



Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:



1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.



2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.



3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.



THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */





contract YAMDelegationStorage {

    /**

     * @notice Implementation address for this contract

     */

    address public implementation;

}



contract YAMDelegatorInterface is YAMDelegationStorage {

    /**

     * @notice Emitted when implementation is changed

     */

    event NewImplementation(address oldImplementation, address newImplementation);



    /**

     * @notice Called by the gov to update the implementation of the delegator

     * @param implementation_ The address of the new implementation for delegation

     * @param allowResign Flag to indicate whether to call _resignImplementation on the old implementation

     * @param becomeImplementationData The encoded bytes data to be passed to _becomeImplementation

     */

    function _setImplementation(address implementation_, bool allowResign, bytes memory becomeImplementationData) public;

}



contract YAMDelegateInterface is YAMDelegationStorage {

    /**

     * @notice Called by the delegator on a delegate to initialize it for duty

     * @dev Should revert if any issues arise which make it unfit for delegation

     * @param data The encoded bytes data for any initialization

     */

    function _becomeImplementation(bytes memory data) public;



    /**

     * @notice Called by the delegator on a delegate to forfeit its responsibility

     */

    function _resignImplementation() public;

}





contract YAMDelegate3 is YAMLogic3, YAMDelegateInterface {

    /**

     * @notice Construct an empty delegate

     */

    constructor() public {}



    /**

     * @notice Called by the delegator on a delegate to initialize it for duty

     * @param data The encoded bytes data for any initialization

     */

    function _becomeImplementation(bytes memory data) public {

        // Shh -- currently unused

        data;



        // Shh -- we don't ever want this hook to be marked pure

        if (false) {

            implementation = address(0);

        }



        require(msg.sender == gov, "only the gov may call _becomeImplementation");

    }



    /**

     * @notice Called by the delegator on a delegate to forfeit its responsibility

     */

    function _resignImplementation() public {

        // Shh -- we don't ever want this hook to be marked pure

        if (false) {

            implementation = address(0);

        }



        require(msg.sender == gov, "only the gov may call _resignImplementation");

    }

}



contract VestingPool {

    using SafeMath for uint256;

    using SafeMath for uint128;



    struct Stream {

        address recipient;

        uint128 startTime;

        uint128 length;

        uint256 totalAmount;

        uint256 amountPaidOut;

    }



    /**

     * @notice Governor for this contract

     */

    address public gov;



    /**

     * @notice Pending governance for this contract

     */

    address public pendingGov;



    /// @notice Mapping containing valid stream managers

    mapping(address => bool) public isSubGov;



    /// @notice Amount of tokens allocated to streams that hasn't yet been claimed

    uint256 public totalUnclaimedInStreams;



    /// @notice The number of streams created so far

    uint256 public streamCount;



    /// @notice All streams

    mapping(uint256 => Stream) public streams;



    /// @notice YAM token

    YAMDelegate3 public yam;



    /**

     * @notice Event emitted when a sub gov is enabled/disabled

     */

    event SubGovModified(

        address account, 

        bool isSubGov

    );



    /**

     * @notice Event emitted when stream is opened

     */

    event StreamOpened(

        address indexed account,

        uint256 indexed streamId,

        uint256 length,

        uint256 totalAmount

    );



    /**

     * @notice Event emitted when stream is closed

     */

    event StreamClosed(uint256 indexed streamId);



    /**

     * @notice Event emitted on payout

     */

    event Payout(

        uint256 indexed streamId,

        address indexed recipient,

        uint256 amount

    );



    /**

     * @notice Event emitted when pendingGov is changed

     */

    event NewPendingGov(

        address oldPendingGov, 

        address newPendingGov

    );



    /**

     * @notice Event emitted when gov is changed

     */

    event NewGov(

        address oldGov, 

        address newGov

    );



    constructor(YAMDelegate3 _yam)

        public

    {

        gov = msg.sender;

        yam = _yam;

    }



    modifier onlyGov() {

        require(msg.sender == gov, "VestingPool::onlyGov: account is not gov");

        _;

    }



    modifier canManageStreams() {

        require(

            isSubGov[msg.sender] || (msg.sender == gov),

            "VestingPool::canManageStreams: account cannot manage streams"

        );

        _;

    }



    /**

     * @dev Set whether an account can open/close streams. Only callable by the current gov contract

     * @param account The account to set permissions for.

     * @param _isSubGov Whether or not this account can manage streams

     */

    function setSubGov(address account, bool _isSubGov)

        public

        onlyGov

    {

        isSubGov[account] = _isSubGov;

        emit SubGovModified(account, _isSubGov);

    }



    /** @notice sets the pendingGov

     * @param pendingGov_ The address of the contract to use for authentication.

     */

    function _setPendingGov(address pendingGov_)

        external

        onlyGov

    {

        address oldPendingGov = pendingGov;

        pendingGov = pendingGov_;

        emit NewPendingGov(oldPendingGov, pendingGov_);

    }



    /** @notice accepts governance over this contract

     *

     */

    function _acceptGov()

        external

    {

        require(msg.sender == pendingGov, "!pending");

        address oldGov = gov;

        gov = pendingGov;

        pendingGov = address(0);

        emit NewGov(oldGov, gov);

    }



    /**

     * @dev Opens a new stream that continuously pays out.

     * @param recipient Account that will receive the funds.

     * @param length The amount of time in seconds that the stream lasts

     * @param totalAmount The total amount to payout in the stream

     */

    function openStream(

        address recipient,

        uint128 length,

        uint256 totalAmount

    )

        public

        canManageStreams

        returns (uint256 streamIndex)

    {

        streamIndex = streamCount++;

        streams[streamIndex] = Stream({

            recipient: recipient,

            length: length,

            startTime: uint128(block.timestamp),

            totalAmount: totalAmount,

            amountPaidOut: 0

        });

        totalUnclaimedInStreams = totalUnclaimedInStreams.add(totalAmount);

        require(

            totalUnclaimedInStreams <= yam.balanceOfUnderlying(address(this)),

            "VestingPool::payout: Total streaming is greater than pool's YAM balance"

        );

        emit StreamOpened(recipient, streamIndex, length, totalAmount);

    }



    /**

     * @dev Closes the specified stream. Pays out pending amounts, clears out the stream, and emits a StreamClosed event.

     * @param streamId The id of the stream to close.

     */

    function closeStream(uint256 streamId)

        public

        canManageStreams

    {

        payout(streamId);

        streams[streamId] = Stream(

            address(0x0000000000000000000000000000000000000000),

            0,

            0,

            0,

            0

        );

        emit StreamClosed(streamId);

    }



    /**

     * @dev Pays out pending amount in a stream

     * @param streamId The id of the stream to payout.

     * @return The amount paid out in underlying

     */

    function payout(uint256 streamId)

        public

        returns (uint256 paidOut)

    {

        uint128 currentTime = uint128(block.timestamp);

        Stream memory stream = streams[streamId];

        require(

            stream.startTime <= currentTime,

            "VestingPool::payout: Stream hasn't started yet"

        );

        uint256 claimableUnderlying = _claimable(stream);

        streams[streamId].amountPaidOut = stream.amountPaidOut.add(

            claimableUnderlying

        );



        totalUnclaimedInStreams = totalUnclaimedInStreams.sub(

            claimableUnderlying

        );



        yam.transferUnderlying(stream.recipient, claimableUnderlying);



        emit Payout(streamId, stream.recipient, claimableUnderlying);

        return claimableUnderlying;

    }





    /**

     * @dev The amount that is claimable for a stream

     * @param streamId The stream to get the claimabout amount for.

     * @return The amount that is claimable for this stream

     */

    function claimable(uint256 streamId)

        external

        view

        returns (uint256 claimableUnderlying)

    {

        Stream memory stream = streams[streamId];

        return _claimable(stream);

    }



    function _claimable(Stream memory stream)

        internal

        view

        returns (uint256 claimableUnderlying)

    {

        uint128 currentTime = uint128(block.timestamp);

        uint128 elapsedTime = currentTime - stream.startTime;

        if (currentTime >= stream.startTime + stream.length) {

            claimableUnderlying = stream.totalAmount - stream.amountPaidOut;

        } else {

            claimableUnderlying = elapsedTime

                .mul(stream.totalAmount)

                .div(stream.length)

                .sub(stream.amountPaidOut);

        }

    }



}





contract MonthlyAllowance {

    using SafeMath for uint256;





    /// @notice Monthly transfer limit - hardcoded to 100,000

    uint256 public constant MONTHLY_LIMIT = 100000 ether;



    /// @notice One month worth of seconds

    uint256 public constant ONE_MONTH = 30 days;



    /// @notice Asset used for payments

    IERC20 public paymentAsset;



    /// @notice Reserves contract to spend from

    address public reserves;



    /// @notice Amount spent per epoch

    mapping(uint256 => uint256) public spentPerEpoch;



    /// @notice Has been initialized

    bool public initialized;



    /// @notice Time initialization happened at - if not initialized, it's 0

    uint256 public timeInitialized;



    /// @notice One way breaker for closing payments from this contract

    bool public breaker;



    /// @notice sub governors

    mapping(address => bool) public isSubGov;



    /// @notice governor

    address public gov;



    /// @notice pending governor

    address public pendingGov;



    /**

     * @notice Event emitted when pendingGov is changed

     */

    event NewPendingGov(

        address oldPendingGov, 

        address newPendingGov

    );



    /**

     * @notice Event emitted when gov is changed

     */

    event NewGov(

        address oldGov, 

        address newGov

    );



    /**

     * @notice Event emitted when a sub gov is enabled/disabled

     */

    event SubGovModified(

        address account, 

        bool isSubGov

    );



    /**

     * @notice Event emitted when a payment is successfully made 

     */

    event Payment(

        address indexed recipient,

        uint256 assetAmount

    );



    modifier onlyGov() {

        require(msg.sender == gov, "MonthlyAllowance::onlyGov: account is not gov");

        _;

    }



    modifier onlyGovOrSubGov() {

        require(msg.sender == gov || isSubGov[msg.sender]);

        _;

    }



    modifier breakerNotSet() {

        require(!breaker, "MonthlyAllowance::breakerNotSet: breaker is set");

        _;

    }

    

    constructor(address _paymentAsset, address _reserves) public {

      gov = msg.sender;

      paymentAsset = IERC20(_paymentAsset);

      reserves = _reserves;

    }



    function initialize()

        public

        onlyGov

    {

        require(!initialized, "MonthlyAllowance::initialize: Contract is already initialized");

        timeInitialized = block.timestamp;

        initialized = true;

    }



    function pay(address recipient, uint256 amount)

        public

        onlyGovOrSubGov

        breakerNotSet

    {

        require(initialized, "MonthlyAllowance::pay: Contract not initialized");

        uint256 epoch = _currentEpoch();

        uint256 newPaidThisEpoch = spentPerEpoch[epoch].add(amount);

        require(newPaidThisEpoch <= MONTHLY_LIMIT, "MonthlyAllowance::pay: Monthly allowance exceeded");

        spentPerEpoch[epoch] = newPaidThisEpoch;

        SafeERC20.safeTransferFrom(paymentAsset, reserves, recipient, amount);

        emit Payment(recipient, amount);

    }



    function currentEpoch()

        public

        returns (uint256)

    {

        return _currentEpoch();

    }



    function _currentEpoch()

        internal

        returns (uint256)

    {

        uint256 timeSinceInitialization = block.timestamp - timeInitialized;

        uint256 epoch = timeSinceInitialization / ONE_MONTH;

        return epoch;

    }



    function flipBreaker()

        public

        onlyGov

        breakerNotSet

    {

        breaker = true;

    }



    function _setPendingGov(address pending)

        public

        onlyGov

    {

        require(pending != address(0));

        address oldPending = pendingGov;

        pendingGov = pending;

        emit NewPendingGov(oldPending, pending);

    }



    function acceptGov()

        public

    {

        require(msg.sender == pendingGov);

        address old = gov;

        gov = pendingGov;

        emit NewGov(old, pendingGov);

    }



    function setIsSubGov(address subGov, bool _isSubGov)

        public

        onlyGov

    {

        isSubGov[subGov] = _isSubGov;

        emit SubGovModified(subGov, _isSubGov);

    }





}



contract StreamManager {

    VestingPool internal constant pool =

        VestingPool(0xDCf613db29E4d0B35e7e15e93BF6cc6315eB0b82);



    MonthlyAllowance internal constant MONTHLY_ALLOWANCE =

        MonthlyAllowance(0x03A882495Bc616D3a1508211312765904Fb062d1);



    function execute() external {

        // Monthly contributors

        MONTHLY_ALLOWANCE.pay(

            0x8A8acf1cEcC4ed6Fe9c408449164CE2034AdC03f,

            yearlyUSDToMonthlyYUSD(120000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0xEC3281124d4c2FCA8A88e3076C1E7749CfEcb7F2,

            yearlyUSDToMonthlyYUSD(105000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0xbdac5657eDd13F47C3DD924eAa36Cf1Ec49672cc,

            yearlyUSDToMonthlyYUSD(72000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0x01e0C7b70E0E05a06c7cC8deeb97Fa03d6a77c9C,

            yearlyUSDToMonthlyYUSD(84000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0xcc506b3c2967022094C3B00276617883167BF32B,

            yearlyUSDToMonthlyYUSD(30000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0x3FdcED6B5C1f176b543E5E0b841cB7224596C33C,

            yearlyUSDToMonthlyYUSD(96000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0xC45d45b54045074Ed12d1Fe127f714f8aCE46f8c,

            yearlyUSDToMonthlyYUSD(45000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0x9098eab0a361D29Ea5c4b5d9d1f50694ac0E9e78,

            yearlyUSDToMonthlyYUSD(36000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0xFcB4f3a1710FefA583e7b003F3165f2E142bC725,

            yearlyUSDToMonthlyYUSD(60000 * (10**18))

        );

        MONTHLY_ALLOWANCE.pay(

            0x31920DF2b31B5f7ecf65BDb2c497DE31d299d472,

            yearlyUSDToMonthlyYUSD(60000 * (10**18))

        );



        // Ronin update

        pool.closeStream(22);

        pool.openStream(

            0xf5f1287F7B71381fFB5Caf3b61fA0375112531BC,

            30 days,

            1250 * 3 * (10**24)

        );



        // Blokku stream update

        pool.closeStream(39);

        pool.openStream(

            0x392027fDc620d397cA27F0c1C3dCB592F27A4dc3,

            30 days,

            256 * 1 * (10**24)

        );



        // Byterose stream update

        pool.closeStream(41);

        pool.openStream(

            0x0Da87C54F853c2CF1221dbE725018944C83BDA7C,

            90 days,

            1280 * 1 * (10**24)

        );



        // Jeremy signing bonus

        pool.payout(

            pool.openStream(

                0xeEFA7451c03d52ce909A93654664c46cf81DdD21,

                0 days,

                800 * (10**24)

            )

        );



        // Tom stream

        pool.openStream(

            0x1b5C706296888a9C52f0c6dcF0579b638bA7EF2a,

            90 days,

            536 * 3 * (10**24)

        );



        // Writer stream

        pool.openStream(

            0x31920DF2b31B5f7ecf65BDb2c497DE31d299d472,

            90 days,

            6664 * 3 * (10**23) // One less decimal since it's 666.4 

        );



        // Close Theo stream

        pool.closeStream(35);



        // Top-up multi-sig

        pool.payout(

            pool.openStream(

                0x744D16d200175d20E6D8e5f405AEfB4EB7A962d1,

                0 days,

                7995767197891679725573970579 // 20k YAM in BoU

            )

        );



        selfdestruct(address(0x0));

    }



    function yearlyUSDToMonthlyYUSD(uint256 yearlyUSD)

        internal

        pure

        returns (uint256)

    {

        // * 100 / 125 accounts for the yUSD price

        return ((yearlyUSD / uint256(12)) * 100) / uint256(129);

    }

}