pragma solidity ^0.4.18;



/**

 * @title ERC20Basic

 * @dev Simpler version of ERC20 interface

 * @dev see https://github.com/ethereum/EIPs/issues/179

 */

contract ERC20Basic {

  uint256 public totalSupply;

  function balanceOf(address who) public view returns (uint256);

  function transfer(address to, uint256 value) public returns (bool);

  event Transfer(address indexed from, address indexed to, uint256 value);

}



/**

 * @title SafeMath

 * @dev Math operations with safety checks that throw on error

 */

library SafeMath {

  function mul(uint256 a, uint256 b) internal pure returns (uint256) {

    if (a == 0) {

      return 0;

    }

    uint256 c = a * b;

    assert(c / a == b);

    return c;

  }



  function div(uint256 a, uint256 b) internal pure returns (uint256) {

    // assert(b > 0); // Solidity automatically throws when dividing by 0

    uint256 c = a / b;

    // assert(a == b * c + a % b); // There is no case in which this doesn't hold

    return c;

  }



  function sub(uint256 a, uint256 b) internal pure returns (uint256) {

    assert(b <= a);

    return a - b;

  }



  function add(uint256 a, uint256 b) internal pure returns (uint256) {

    uint256 c = a + b;

    assert(c >= a);

    return c;

  }

}



/**

 * @title Basic token

 * @dev Basic version of StandardToken, with no allowances.

 */

contract BasicToken is ERC20Basic {

  using SafeMath for uint256;



  mapping(address => uint256) balances;



  /**

  * @dev transfer token for a specified address

  * @param _to The address to transfer to.

  * @param _value The amount to be transferred.

  */

  function transfer(address _to, uint256 _value) public returns (bool) {

    require(_to != address(0));

    require(_value <= balances[msg.sender]);



    // SafeMath.sub will throw if there is not enough balance.

    balances[msg.sender] = balances[msg.sender].sub(_value);

    balances[_to] = balances[_to].add(_value);

    Transfer(msg.sender, _to, _value);

    return true;

  }



  /**

  * @dev Gets the balance of the specified address.

  * @param _owner The address to query the the balance of.

  * @return An uint256 representing the amount owned by the passed address.

  */

  function balanceOf(address _owner) public view returns (uint256 balance) {

    return balances[_owner];

  }



}



/**

 * @title ERC20 interface

 * @dev see https://github.com/ethereum/EIPs/issues/20

 */

contract ERC20 is ERC20Basic {

  function allowance(address owner, address spender) public view returns (uint256);

  function transferFrom(address from, address to, uint256 value) public returns (bool);

  function approve(address spender, uint256 value) public returns (bool);

  event Approval(address indexed owner, address indexed spender, uint256 value);

}



/**

 * @title Standard ERC20 token

 *

 * @dev Implementation of the basic standard token.

 * @dev https://github.com/ethereum/EIPs/issues/20

 * @dev Based on code by FirstBlood: https://github.com/Firstbloodio/token/blob/master/smart_contract/FirstBloodToken.sol

 */

contract StandardToken is ERC20, BasicToken {



  mapping (address => mapping (address => uint256)) internal allowed;





  /**

   * @dev Transfer tokens from one address to another

   * @param _from address The address which you want to send tokens from

   * @param _to address The address which you want to transfer to

   * @param _value uint256 the amount of tokens to be transferred

   */

  function transferFrom(address _from, address _to, uint256 _value) public returns (bool) {

    require(_to != address(0));

    require(_value <= balances[_from]);

    require(_value <= allowed[_from][msg.sender]);



    balances[_from] = balances[_from].sub(_value);

    balances[_to] = balances[_to].add(_value);

    allowed[_from][msg.sender] = allowed[_from][msg.sender].sub(_value);

    Transfer(_from, _to, _value);

    return true;

  }



  /**

   * @dev Approve the passed address to spend the specified amount of tokens on behalf of msg.sender.

   *

   * Beware that changing an allowance with this method brings the risk that someone may use both the old

   * and the new allowance by unfortunate transaction ordering. One possible solution to mitigate this

   * race condition is to first reduce the spender's allowance to 0 and set the desired value afterwards:

   * https://github.com/ethereum/EIPs/issues/20#issuecomment-263524729

   * @param _spender The address which will spend the funds.

   * @param _value The amount of tokens to be spent.

   */

  function approve(address _spender, uint256 _value) public returns (bool) {

    allowed[msg.sender][_spender] = _value;

    Approval(msg.sender, _spender, _value);

    return true;

  }



  /**

   * @dev Function to check the amount of tokens that an owner allowed to a spender.

   * @param _owner address The address which owns the funds.

   * @param _spender address The address which will spend the funds.

   * @return A uint256 specifying the amount of tokens still available for the spender.

   */

  function allowance(address _owner, address _spender) public view returns (uint256) {

    return allowed[_owner][_spender];

  }



  /**

   * @dev Increase the amount of tokens that an owner allowed to a spender.

   *

   * approve should be called when allowed[_spender] == 0. To increment

   * allowed value is better to use this function to avoid 2 calls (and wait until

   * the first transaction is mined)

   * From MonolithDAO Token.sol

   * @param _spender The address which will spend the funds.

   * @param _addedValue The amount of tokens to increase the allowance by.

   */

  function increaseApproval(address _spender, uint _addedValue) public returns (bool) {

    allowed[msg.sender][_spender] = allowed[msg.sender][_spender].add(_addedValue);

    Approval(msg.sender, _spender, allowed[msg.sender][_spender]);

    return true;

  }



  /**

   * @dev Decrease the amount of tokens that an owner allowed to a spender.

   *

   * approve should be called when allowed[_spender] == 0. To decrement

   * allowed value is better to use this function to avoid 2 calls (and wait until

   * the first transaction is mined)

   * From MonolithDAO Token.sol

   * @param _spender The address which will spend the funds.

   * @param _subtractedValue The amount of tokens to decrease the allowance by.

   */

  function decreaseApproval(address _spender, uint _subtractedValue) public returns (bool) {

    uint oldValue = allowed[msg.sender][_spender];

    if (_subtractedValue > oldValue) {

      allowed[msg.sender][_spender] = 0;

    } else {

      allowed[msg.sender][_spender] = oldValue.sub(_subtractedValue);

    }

    Approval(msg.sender, _spender, allowed[msg.sender][_spender]);

    return true;

  }



}



/**

 * @title Ownable

 * @dev The Ownable contract has an owner address, and provides basic authorization control

 * functions, this simplifies the implementation of "user permissions".

 */

contract Ownable {

  address public owner;





  event OwnershipTransferred(address indexed previousOwner, address indexed newOwner);





  /**

   * @dev The Ownable constructor sets the original `owner` of the contract to the sender

   * account.

   */

  function Ownable() public {

    owner = msg.sender;

  }





  /**

   * @dev Throws if called by any account other than the owner.

   */

  modifier onlyOwner() {

    require(msg.sender == owner);

    _;

  }





  /**

   * @dev Allows the current owner to transfer control of the contract to a newOwner.

   * @param newOwner The address to transfer ownership to.

   */

  function transferOwnership(address newOwner) public onlyOwner {

    require(newOwner != address(0));

    OwnershipTransferred(owner, newOwner);

    owner = newOwner;

  }



}



/**

 * @title Mintable token

 * @dev Simple ERC20 Token example, with mintable token creation

 * @dev Issue: * https://github.com/OpenZeppelin/zeppelin-solidity/issues/120

 * Based on code by TokenMarketNet: https://github.com/TokenMarketNet/ico/blob/master/contracts/MintableToken.sol

 */



contract MintableToken is StandardToken, Ownable {

  event Mint(address indexed to, uint256 amount);

  event MintFinished();



  bool public mintingFinished = false;





  modifier canMint() {

    require(!mintingFinished);

    _;

  }



  /**

   * @dev Function to mint tokens

   * @param _to The address that will receive the minted tokens.

   * @param _amount The amount of tokens to mint.

   * @return A boolean that indicates if the operation was successful.

   */

  function mint(address _to, uint256 _amount) onlyOwner canMint public returns (bool) {

    totalSupply = totalSupply.add(_amount);

    balances[_to] = balances[_to].add(_amount);

    Mint(_to, _amount);

    Transfer(address(0), _to, _amount);

    return true;

  }



  /**

   * @dev Function to stop minting new tokens.

   * @return True if the operation was successful.

   */

  function finishMinting() onlyOwner canMint public returns (bool) {

    mintingFinished = true;

    MintFinished();

    return true;

  }

}



/**

 * @title Crowdsale

 * @dev Crowdsale is a base contract for managing a token crowdsale.

 * Crowdsales have a start and end timestamps, where investors can make

 * token purchases and the crowdsale will assign them tokens based

 * on a token per ETH rate. Funds collected are forwarded to a wallet

 * as they arrive.

 */

contract Crowdsale {

  using SafeMath for uint256;



  // The token being sold

  MintableToken public token;



  // start and end timestamps where investments are allowed (both inclusive)

  uint256 public startTime;

  uint256 public endTime;



  // address where funds are collected

  address public wallet;



  // how many token units a buyer gets per wei

  uint256 public rate;



  // amount of raised money in wei

  uint256 public weiRaised;



  /**

   * event for token purchase logging

   * @param purchaser who paid for the tokens

   * @param beneficiary who got the tokens

   * @param value weis paid for purchase

   * @param amount amount of tokens purchased

   */

  event TokenPurchase(address indexed purchaser, address indexed beneficiary, uint256 value, uint256 amount);





  function Crowdsale(uint256 _startTime, uint256 _endTime, uint256 _rate, address _wallet) public {

    require(_startTime >= now);

    require(_endTime >= _startTime);

    require(_rate > 0);

    require(_wallet != address(0));



    token = createTokenContract();

    startTime = _startTime;

    endTime = _endTime;

    rate = _rate;

    wallet = _wallet;

  }



  // creates the token to be sold.

  // override this method to have crowdsale of a specific mintable token.

  function createTokenContract() internal returns (MintableToken) {

    return new MintableToken();

  }





  // fallback function can be used to buy tokens

  function () external payable {

    buyTokens(msg.sender);

  }



  // low level token purchase function

  function buyTokens(address beneficiary) public payable {

    require(beneficiary != address(0));

    require(validPurchase());



    uint256 weiAmount = msg.value;



    // calculate token amount to be created

    uint256 tokens = weiAmount.mul(rate);



    // update state

    weiRaised = weiRaised.add(weiAmount);



    token.mint(beneficiary, tokens);

    TokenPurchase(msg.sender, beneficiary, weiAmount, tokens);



    forwardFunds();

  }



  // send ether to the fund collection wallet

  // override to create custom fund forwarding mechanisms

  function forwardFunds() internal {

    wallet.transfer(msg.value);

  }



  // @return true if the transaction can buy tokens

  function validPurchase() internal view returns (bool) {

    bool withinPeriod = now >= startTime && now <= endTime;

    bool nonZeroPurchase = msg.value != 0;

    return withinPeriod && nonZeroPurchase;

  }



  // @return true if crowdsale event has ended

  function hasEnded() public view returns (bool) {

    return now > endTime;

  }





}



contract DetailedERC20 is ERC20 {

  string public name;

  string public symbol;

  uint8 public decimals;



  function DetailedERC20(string _name, string _symbol, uint8 _decimals) public {

    name = _name;

    symbol = _symbol;

    decimals = _decimals;

  }

}



/**

 * @title Capped token

 * @dev Mintable token with a token cap.

 */



contract CappedToken is MintableToken {



  uint256 public cap;



  function CappedToken(uint256 _cap) public {

    require(_cap > 0);

    cap = _cap;

  }



  /**

   * @dev Function to mint tokens

   * @param _to The address that will receive the minted tokens.

   * @param _amount The amount of tokens to mint.

   * @return A boolean that indicates if the operation was successful.

   */

  function mint(address _to, uint256 _amount) onlyOwner canMint public returns (bool) {

    require(totalSupply.add(_amount) <= cap);



    return super.mint(_to, _amount);

  }



}



/**

 * @title Burnable Token

 * @dev Token that can be irreversibly burned (destroyed).

 */

contract BurnableToken is BasicToken {



    event Burn(address indexed burner, uint256 value);



    /**

     * @dev Burns a specific amount of tokens.

     * @param _value The amount of token to be burned.

     */

    function burn(uint256 _value) public {

        require(_value <= balances[msg.sender]);

        // no need to require value <= totalSupply, since that would imply the

        // sender's balance is greater than the totalSupply, which *should* be an assertion failure



        address burner = msg.sender;

        balances[burner] = balances[burner].sub(_value);

        totalSupply = totalSupply.sub(_value);

        Burn(burner, _value);

    }

}



/**

 * @title SafeERC20

 * @dev Wrappers around ERC20 operations that throw on failure.

 * To use this library you can add a `using SafeERC20 for ERC20;` statement to your contract,

 * which allows you to call the safe operations as `token.safeTransfer(...)`, etc.

 */

library SafeERC20 {

  function safeTransfer(ERC20Basic token, address to, uint256 value) internal {

    assert(token.transfer(to, value));

  }



  function safeTransferFrom(ERC20 token, address from, address to, uint256 value) internal {

    assert(token.transferFrom(from, to, value));

  }



  function safeApprove(ERC20 token, address spender, uint256 value) internal {

    assert(token.approve(spender, value));

  }

}



//it mimic the TokenTimelock



contract HTLC {



    using SafeERC20 for ERC20Basic;



    // ERC20 basic token contract being held

    ERC20Basic public token;



    // beneficiary of tokens being released by hashing

    address public beneficiary;



    // timestamp when lock is timeout

    uint256 public releaseTime;



    // sha256 hash to release

    bytes32 sha256hash;



    function HTLC(ERC20Basic _token, bytes32 _hash, address _beneficiary, uint256 _releaseTime) public {

        require(_releaseTime > now);

        token = _token;

        beneficiary = _beneficiary;

        releaseTime = _releaseTime;

        sha256hash = _hash;

    }



    /**

     * @notice Transfer to the specified address if anyone can provide the correct preimage.

     */

    function redeem(bytes preimage) public {

        require(sha256(preimage) == sha256hash);



        uint256 amount = token.balanceOf(this);

        require(amount > 0);



        token.safeTransfer(beneficiary, amount);

        selfdestruct(msg.sender);

    }



    /** 

     * @notice Default action when timeout: override this.

     */

    function onTimeout(uint256) internal {

        selfdestruct(msg.sender);

    }



    /**

     * @notice When timeout. for default action, the token in it will lost forever

     */

    function release() public {

        require(now >= releaseTime);



        uint256 amount = token.balanceOf(this);

        require(amount > 0);



        onTimeout(amount);

    }



}



contract VCBToken is CappedToken, BurnableToken, DetailedERC20 {



    using SafeMath for uint256;



    uint8 constant DECIMALS = 18;

    uint  constant TOTALTOKEN = 1 * 10 ** (8 + uint(DECIMALS));

    string constant NAME = "ValueCyber Token";

    string constant SYM = "VCT";



    address constant PRESALE = 0x638a3C7dF9D1B3A56E19B92bE07eCC84b6475BD6;

    uint  constant PRESALETOKEN = 55 * 10 ** (6 + uint(DECIMALS));



    function VCBToken() CappedToken(TOTALTOKEN) DetailedERC20 (NAME, SYM, DECIMALS) public {

        

        balances[PRESALE] = PRESALETOKEN;

        totalSupply = totalSupply.add(PRESALETOKEN);

    }



}



contract VCBCrowdSale is Crowdsale, Ownable {



    using SafeMath for uint256;



    uint  constant RATIO = 800;

    uint8 constant RATIODENO = 100;

    uint constant SALELASTFOR = 50 days;

    address constant FUNDWALLET = 0x622969e0928fa6bEeda9f26F8a60D0b22Db7E6f1;



    mapping(address => uint8) giftList;



    event CrowdsaleFinalized();

    /**

    * event for token gift logging

    * @param beneficiary who got the gifted tokens

    * @param amount amount of tokens gifted

    */

    event TokenGift(address indexed beneficiary, uint256 amount);



    function VCBCrowdSale() Crowdsale(now, now + SALELASTFOR, RATIO, FUNDWALLET) public {

    }



    function createTokenContract() internal returns (MintableToken) {

        return new VCBToken();

    }



    //our crowdsale can stop at anytime, and then the totally crowsale contract is disappear

    function finalize(address _finaladdr) onlyOwner public {

        token.finishMinting();

        CrowdsaleFinalized();



        address finaladdr = FUNDWALLET;

        if (_finaladdr != address(0)) {

            finaladdr = _finaladdr;

        }



        selfdestruct(finaladdr);

    }  



    function giftTokens(address beneficiary) internal {

        uint256 weiAmount = msg.value;



        // calculate token amount to be created

        uint256 gifttokens = weiAmount.mul(giftList[beneficiary]).mul(rate).div(RATIODENO);

        if (gifttokens > 0) {



            //if gift token can't be sent, contract still fails

            token.mint(beneficiary, gifttokens);

            TokenGift(beneficiary, gifttokens);

        }



    }



    // override token purchase to send additional token for registered address

    function buyTokens(address beneficiary) public payable {



        super.buyTokens(beneficiary);



        //if address is in discount list, we gift it more tokens according to the ratio (in percentage)

        giftTokens(beneficiary);

    }



    function addGift(address beneficiary, uint8 giftratio) onlyOwner public {

        require(giftratio < RATIODENO);

        giftList[beneficiary] = giftratio;

    }



    /**

    * @dev Gets the gift ratio of the specified address.

    * @param _owner The address to query the gift ratio of.

    * @return An uint8 representing the ratio obtained by the passed address.

    */

    function giftRatioOf(address _owner) public view returns (uint8 ratio) {

        return giftList[_owner];

    }



    // directly mint tokens to a HTLC contract

    function preserveTokens(address preservecontract, uint256 amount) onlyOwner public {        

        token.mint(preservecontract, amount);

    }    



}



contract CrowdSaleHTLC is HTLC {

    function CrowdSaleHTLC(ERC20Basic _token, bytes32 _hash, address _beneficiary, uint256 _releaseTime) HTLC(_token, _hash, _beneficiary, _releaseTime) public {

    }



    function onTimeout(uint256 amount) internal {

        BurnableToken t = BurnableToken (token);

        t.burn(amount);

        super.onTimeout(amount);

    }



}