pragma solidity ^0.4.24;



// File: zeppelin-solidity/contracts/introspection/ERC165.sol



/**

 * @title ERC165

 * @dev https://github.com/ethereum/EIPs/blob/master/EIPS/eip-165.md

 */

interface ERC165 {



  /**

   * @notice Query if a contract implements an interface

   * @param _interfaceId The interface identifier, as specified in ERC-165

   * @dev Interface identification is specified in ERC-165. This function

   * uses less than 30,000 gas.

   */

  function supportsInterface(bytes4 _interfaceId)

    external

    view

    returns (bool);

}



// File: zeppelin-solidity/contracts/token/ERC721/ERC721Basic.sol



/**

 * @title ERC721 Non-Fungible Token Standard basic interface

 * @dev see https://github.com/ethereum/EIPs/blob/master/EIPS/eip-721.md

 */

contract ERC721Basic is ERC165 {



  bytes4 internal constant InterfaceId_ERC721 = 0x80ac58cd;

  /*

   * 0x80ac58cd ===

   *   bytes4(keccak256('balanceOf(address)')) ^

   *   bytes4(keccak256('ownerOf(uint256)')) ^

   *   bytes4(keccak256('approve(address,uint256)')) ^

   *   bytes4(keccak256('getApproved(uint256)')) ^

   *   bytes4(keccak256('setApprovalForAll(address,bool)')) ^

   *   bytes4(keccak256('isApprovedForAll(address,address)')) ^

   *   bytes4(keccak256('transferFrom(address,address,uint256)')) ^

   *   bytes4(keccak256('safeTransferFrom(address,address,uint256)')) ^

   *   bytes4(keccak256('safeTransferFrom(address,address,uint256,bytes)'))

   */



  bytes4 internal constant InterfaceId_ERC721Exists = 0x4f558e79;

  /*

   * 0x4f558e79 ===

   *   bytes4(keccak256('exists(uint256)'))

   */



  bytes4 internal constant InterfaceId_ERC721Enumerable = 0x780e9d63;

  /**

   * 0x780e9d63 ===

   *   bytes4(keccak256('totalSupply()')) ^

   *   bytes4(keccak256('tokenOfOwnerByIndex(address,uint256)')) ^

   *   bytes4(keccak256('tokenByIndex(uint256)'))

   */



  bytes4 internal constant InterfaceId_ERC721Metadata = 0x5b5e139f;

  /**

   * 0x5b5e139f ===

   *   bytes4(keccak256('name()')) ^

   *   bytes4(keccak256('symbol()')) ^

   *   bytes4(keccak256('tokenURI(uint256)'))

   */



  event Transfer(

    address indexed _from,

    address indexed _to,

    uint256 indexed _tokenId

  );

  event Approval(

    address indexed _owner,

    address indexed _approved,

    uint256 indexed _tokenId

  );

  event ApprovalForAll(

    address indexed _owner,

    address indexed _operator,

    bool _approved

  );



  function balanceOf(address _owner) public view returns (uint256 _balance);

  function ownerOf(uint256 _tokenId) public view returns (address _owner);

  function exists(uint256 _tokenId) public view returns (bool _exists);



  function approve(address _to, uint256 _tokenId) public;

  function getApproved(uint256 _tokenId)

    public view returns (address _operator);



  function setApprovalForAll(address _operator, bool _approved) public;

  function isApprovedForAll(address _owner, address _operator)

    public view returns (bool);



  function transferFrom(address _from, address _to, uint256 _tokenId) public;

  function safeTransferFrom(address _from, address _to, uint256 _tokenId)

    public;



  function safeTransferFrom(

    address _from,

    address _to,

    uint256 _tokenId,

    bytes _data

  )

    public;

}



// File: zeppelin-solidity/contracts/token/ERC721/ERC721.sol



/**

 * @title ERC-721 Non-Fungible Token Standard, optional enumeration extension

 * @dev See https://github.com/ethereum/EIPs/blob/master/EIPS/eip-721.md

 */

contract ERC721Enumerable is ERC721Basic {

  function totalSupply() public view returns (uint256);

  function tokenOfOwnerByIndex(

    address _owner,

    uint256 _index

  )

    public

    view

    returns (uint256 _tokenId);



  function tokenByIndex(uint256 _index) public view returns (uint256);

}





/**

 * @title ERC-721 Non-Fungible Token Standard, optional metadata extension

 * @dev See https://github.com/ethereum/EIPs/blob/master/EIPS/eip-721.md

 */

contract ERC721Metadata is ERC721Basic {

  function name() external view returns (string _name);

  function symbol() external view returns (string _symbol);

  function tokenURI(uint256 _tokenId) public view returns (string);

}





/**

 * @title ERC-721 Non-Fungible Token Standard, full implementation interface

 * @dev See https://github.com/ethereum/EIPs/blob/master/EIPS/eip-721.md

 */

contract ERC721 is ERC721Basic, ERC721Enumerable, ERC721Metadata {

}



// File: zeppelin-solidity/contracts/token/ERC721/ERC721Receiver.sol



/**

 * @title ERC721 token receiver interface

 * @dev Interface for any contract that wants to support safeTransfers

 * from ERC721 asset contracts.

 */

contract ERC721Receiver {

  /**

   * @dev Magic value to be returned upon successful reception of an NFT

   *  Equals to `bytes4(keccak256("onERC721Received(address,address,uint256,bytes)"))`,

   *  which can be also obtained as `ERC721Receiver(0).onERC721Received.selector`

   */

  bytes4 internal constant ERC721_RECEIVED = 0x150b7a02;



  /**

   * @notice Handle the receipt of an NFT

   * @dev The ERC721 smart contract calls this function on the recipient

   * after a `safetransfer`. This function MAY throw to revert and reject the

   * transfer. Return of other than the magic value MUST result in the

   * transaction being reverted.

   * Note: the contract address is always the message sender.

   * @param _operator The address which called `safeTransferFrom` function

   * @param _from The address which previously owned the token

   * @param _tokenId The NFT identifier which is being transferred

   * @param _data Additional data with no specified format

   * @return `bytes4(keccak256("onERC721Received(address,address,uint256,bytes)"))`

   */

  function onERC721Received(

    address _operator,

    address _from,

    uint256 _tokenId,

    bytes _data

  )

    public

    returns(bytes4);

}



// File: zeppelin-solidity/contracts/math/SafeMath.sol



/**

 * @title SafeMath

 * @dev Math operations with safety checks that throw on error

 */

library SafeMath {



  /**

  * @dev Multiplies two numbers, throws on overflow.

  */

  function mul(uint256 _a, uint256 _b) internal pure returns (uint256 c) {

    // Gas optimization: this is cheaper than asserting 'a' not being zero, but the

    // benefit is lost if 'b' is also tested.

    // See: https://github.com/OpenZeppelin/openzeppelin-solidity/pull/522

    if (_a == 0) {

      return 0;

    }



    c = _a * _b;

    assert(c / _a == _b);

    return c;

  }



  /**

  * @dev Integer division of two numbers, truncating the quotient.

  */

  function div(uint256 _a, uint256 _b) internal pure returns (uint256) {

    // assert(_b > 0); // Solidity automatically throws when dividing by 0

    // uint256 c = _a / _b;

    // assert(_a == _b * c + _a % _b); // There is no case in which this doesn't hold

    return _a / _b;

  }



  /**

  * @dev Subtracts two numbers, throws on overflow (i.e. if subtrahend is greater than minuend).

  */

  function sub(uint256 _a, uint256 _b) internal pure returns (uint256) {

    assert(_b <= _a);

    return _a - _b;

  }



  /**

  * @dev Adds two numbers, throws on overflow.

  */

  function add(uint256 _a, uint256 _b) internal pure returns (uint256 c) {

    c = _a + _b;

    assert(c >= _a);

    return c;

  }

}



// File: zeppelin-solidity/contracts/AddressUtils.sol



/**

 * Utility library of inline functions on addresses

 */

library AddressUtils {



  /**

   * Returns whether the target address is a contract

   * @dev This function will return false if invoked during the constructor of a contract,

   * as the code is not actually created until after the constructor finishes.

   * @param _addr address to check

   * @return whether the target address is a contract

   */

  function isContract(address _addr) internal view returns (bool) {

    uint256 size;

    // XXX Currently there is no better way to check if there is a contract in an address

    // than to check the size of the code at that address.

    // See https://ethereum.stackexchange.com/a/14016/36603

    // for more details about how this works.

    // TODO Check this again before the Serenity release, because all addresses will be

    // contracts then.

    // solium-disable-next-line security/no-inline-assembly

    assembly { size := extcodesize(_addr) }

    return size > 0;

  }



}



// File: zeppelin-solidity/contracts/introspection/SupportsInterfaceWithLookup.sol



/**

 * @title SupportsInterfaceWithLookup

 * @author Matt Condon (@shrugs)

 * @dev Implements ERC165 using a lookup table.

 */

contract SupportsInterfaceWithLookup is ERC165 {



  bytes4 public constant InterfaceId_ERC165 = 0x01ffc9a7;

  /**

   * 0x01ffc9a7 ===

   *   bytes4(keccak256('supportsInterface(bytes4)'))

   */



  /**

   * @dev a mapping of interface id to whether or not it's supported

   */

  mapping(bytes4 => bool) internal supportedInterfaces;



  /**

   * @dev A contract implementing SupportsInterfaceWithLookup

   * implement ERC165 itself

   */

  constructor()

    public

  {

    _registerInterface(InterfaceId_ERC165);

  }



  /**

   * @dev implement supportsInterface(bytes4) using a lookup table

   */

  function supportsInterface(bytes4 _interfaceId)

    external

    view

    returns (bool)

  {

    return supportedInterfaces[_interfaceId];

  }



  /**

   * @dev private method for registering an interface

   */

  function _registerInterface(bytes4 _interfaceId)

    internal

  {

    require(_interfaceId != 0xffffffff);

    supportedInterfaces[_interfaceId] = true;

  }

}



// File: zeppelin-solidity/contracts/token/ERC721/ERC721BasicToken.sol



/**

 * @title ERC721 Non-Fungible Token Standard basic implementation

 * @dev see https://github.com/ethereum/EIPs/blob/master/EIPS/eip-721.md

 */

contract ERC721BasicToken is SupportsInterfaceWithLookup, ERC721Basic {



  using SafeMath for uint256;

  using AddressUtils for address;



  // Equals to `bytes4(keccak256("onERC721Received(address,address,uint256,bytes)"))`

  // which can be also obtained as `ERC721Receiver(0).onERC721Received.selector`

  bytes4 private constant ERC721_RECEIVED = 0x150b7a02;



  // Mapping from token ID to owner

  mapping (uint256 => address) internal tokenOwner;



  // Mapping from token ID to approved address

  mapping (uint256 => address) internal tokenApprovals;



  // Mapping from owner to number of owned token

  mapping (address => uint256) internal ownedTokensCount;



  // Mapping from owner to operator approvals

  mapping (address => mapping (address => bool)) internal operatorApprovals;



  constructor()

    public

  {

    // register the supported interfaces to conform to ERC721 via ERC165

    _registerInterface(InterfaceId_ERC721);

    _registerInterface(InterfaceId_ERC721Exists);

  }



  /**

   * @dev Gets the balance of the specified address

   * @param _owner address to query the balance of

   * @return uint256 representing the amount owned by the passed address

   */

  function balanceOf(address _owner) public view returns (uint256) {

    require(_owner != address(0));

    return ownedTokensCount[_owner];

  }



  /**

   * @dev Gets the owner of the specified token ID

   * @param _tokenId uint256 ID of the token to query the owner of

   * @return owner address currently marked as the owner of the given token ID

   */

  function ownerOf(uint256 _tokenId) public view returns (address) {

    address owner = tokenOwner[_tokenId];

    require(owner != address(0));

    return owner;

  }



  /**

   * @dev Returns whether the specified token exists

   * @param _tokenId uint256 ID of the token to query the existence of

   * @return whether the token exists

   */

  function exists(uint256 _tokenId) public view returns (bool) {

    address owner = tokenOwner[_tokenId];

    return owner != address(0);

  }



  /**

   * @dev Approves another address to transfer the given token ID

   * The zero address indicates there is no approved address.

   * There can only be one approved address per token at a given time.

   * Can only be called by the token owner or an approved operator.

   * @param _to address to be approved for the given token ID

   * @param _tokenId uint256 ID of the token to be approved

   */

  function approve(address _to, uint256 _tokenId) public {

    address owner = ownerOf(_tokenId);

    require(_to != owner);

    require(msg.sender == owner || isApprovedForAll(owner, msg.sender));



    tokenApprovals[_tokenId] = _to;

    emit Approval(owner, _to, _tokenId);

  }



  /**

   * @dev Gets the approved address for a token ID, or zero if no address set

   * @param _tokenId uint256 ID of the token to query the approval of

   * @return address currently approved for the given token ID

   */

  function getApproved(uint256 _tokenId) public view returns (address) {

    return tokenApprovals[_tokenId];

  }



  /**

   * @dev Sets or unsets the approval of a given operator

   * An operator is allowed to transfer all tokens of the sender on their behalf

   * @param _to operator address to set the approval

   * @param _approved representing the status of the approval to be set

   */

  function setApprovalForAll(address _to, bool _approved) public {

    require(_to != msg.sender);

    operatorApprovals[msg.sender][_to] = _approved;

    emit ApprovalForAll(msg.sender, _to, _approved);

  }



  /**

   * @dev Tells whether an operator is approved by a given owner

   * @param _owner owner address which you want to query the approval of

   * @param _operator operator address which you want to query the approval of

   * @return bool whether the given operator is approved by the given owner

   */

  function isApprovedForAll(

    address _owner,

    address _operator

  )

    public

    view

    returns (bool)

  {

    return operatorApprovals[_owner][_operator];

  }



  /**

   * @dev Transfers the ownership of a given token ID to another address

   * Usage of this method is discouraged, use `safeTransferFrom` whenever possible

   * Requires the msg sender to be the owner, approved, or operator

   * @param _from current owner of the token

   * @param _to address to receive the ownership of the given token ID

   * @param _tokenId uint256 ID of the token to be transferred

  */

  function transferFrom(

    address _from,

    address _to,

    uint256 _tokenId

  )

    public

  {

    require(isApprovedOrOwner(msg.sender, _tokenId));

    require(_from != address(0));

    require(_to != address(0));



    clearApproval(_from, _tokenId);

    removeTokenFrom(_from, _tokenId);

    addTokenTo(_to, _tokenId);



    emit Transfer(_from, _to, _tokenId);

  }



  /**

   * @dev Safely transfers the ownership of a given token ID to another address

   * If the target address is a contract, it must implement `onERC721Received`,

   * which is called upon a safe transfer, and return the magic value

   * `bytes4(keccak256("onERC721Received(address,address,uint256,bytes)"))`; otherwise,

   * the transfer is reverted.

   *

   * Requires the msg sender to be the owner, approved, or operator

   * @param _from current owner of the token

   * @param _to address to receive the ownership of the given token ID

   * @param _tokenId uint256 ID of the token to be transferred

  */

  function safeTransferFrom(

    address _from,

    address _to,

    uint256 _tokenId

  )

    public

  {

    // solium-disable-next-line arg-overflow

    safeTransferFrom(_from, _to, _tokenId, "");

  }



  /**

   * @dev Safely transfers the ownership of a given token ID to another address

   * If the target address is a contract, it must implement `onERC721Received`,

   * which is called upon a safe transfer, and return the magic value

   * `bytes4(keccak256("onERC721Received(address,address,uint256,bytes)"))`; otherwise,

   * the transfer is reverted.

   * Requires the msg sender to be the owner, approved, or operator

   * @param _from current owner of the token

   * @param _to address to receive the ownership of the given token ID

   * @param _tokenId uint256 ID of the token to be transferred

   * @param _data bytes data to send along with a safe transfer check

   */

  function safeTransferFrom(

    address _from,

    address _to,

    uint256 _tokenId,

    bytes _data

  )

    public

  {

    transferFrom(_from, _to, _tokenId);

    // solium-disable-next-line arg-overflow

    require(checkAndCallSafeTransfer(_from, _to, _tokenId, _data));

  }



  /**

   * @dev Returns whether the given spender can transfer a given token ID

   * @param _spender address of the spender to query

   * @param _tokenId uint256 ID of the token to be transferred

   * @return bool whether the msg.sender is approved for the given token ID,

   *  is an operator of the owner, or is the owner of the token

   */

  function isApprovedOrOwner(

    address _spender,

    uint256 _tokenId

  )

    internal

    view

    returns (bool)

  {

    address owner = ownerOf(_tokenId);

    // Disable solium check because of

    // https://github.com/duaraghav8/Solium/issues/175

    // solium-disable-next-line operator-whitespace

    return (

      _spender == owner ||

      getApproved(_tokenId) == _spender ||

      isApprovedForAll(owner, _spender)

    );

  }



  /**

   * @dev Internal function to mint a new token

   * Reverts if the given token ID already exists

   * @param _to The address that will own the minted token

   * @param _tokenId uint256 ID of the token to be minted by the msg.sender

   */

  function _mint(address _to, uint256 _tokenId) internal {

    require(_to != address(0));

    addTokenTo(_to, _tokenId);

    emit Transfer(address(0), _to, _tokenId);

  }



  /**

   * @dev Internal function to burn a specific token

   * Reverts if the token does not exist

   * @param _tokenId uint256 ID of the token being burned by the msg.sender

   */

  function _burn(address _owner, uint256 _tokenId) internal {

    clearApproval(_owner, _tokenId);

    removeTokenFrom(_owner, _tokenId);

    emit Transfer(_owner, address(0), _tokenId);

  }



  /**

   * @dev Internal function to clear current approval of a given token ID

   * Reverts if the given address is not indeed the owner of the token

   * @param _owner owner of the token

   * @param _tokenId uint256 ID of the token to be transferred

   */

  function clearApproval(address _owner, uint256 _tokenId) internal {

    require(ownerOf(_tokenId) == _owner);

    if (tokenApprovals[_tokenId] != address(0)) {

      tokenApprovals[_tokenId] = address(0);

    }

  }



  /**

   * @dev Internal function to add a token ID to the list of a given address

   * @param _to address representing the new owner of the given token ID

   * @param _tokenId uint256 ID of the token to be added to the tokens list of the given address

   */

  function addTokenTo(address _to, uint256 _tokenId) internal {

    require(tokenOwner[_tokenId] == address(0));

    tokenOwner[_tokenId] = _to;

    ownedTokensCount[_to] = ownedTokensCount[_to].add(1);

  }



  /**

   * @dev Internal function to remove a token ID from the list of a given address

   * @param _from address representing the previous owner of the given token ID

   * @param _tokenId uint256 ID of the token to be removed from the tokens list of the given address

   */

  function removeTokenFrom(address _from, uint256 _tokenId) internal {

    require(ownerOf(_tokenId) == _from);

    ownedTokensCount[_from] = ownedTokensCount[_from].sub(1);

    tokenOwner[_tokenId] = address(0);

  }



  /**

   * @dev Internal function to invoke `onERC721Received` on a target address

   * The call is not executed if the target address is not a contract

   * @param _from address representing the previous owner of the given token ID

   * @param _to target address that will receive the tokens

   * @param _tokenId uint256 ID of the token to be transferred

   * @param _data bytes optional data to send along with the call

   * @return whether the call correctly returned the expected magic value

   */

  function checkAndCallSafeTransfer(

    address _from,

    address _to,

    uint256 _tokenId,

    bytes _data

  )

    internal

    returns (bool)

  {

    if (!_to.isContract()) {

      return true;

    }

    bytes4 retval = ERC721Receiver(_to).onERC721Received(

      msg.sender, _from, _tokenId, _data);

    return (retval == ERC721_RECEIVED);

  }

}



// File: zeppelin-solidity/contracts/token/ERC721/ERC721Token.sol



/**

 * @title Full ERC721 Token

 * This implementation includes all the required and some optional functionality of the ERC721 standard

 * Moreover, it includes approve all functionality using operator terminology

 * @dev see https://github.com/ethereum/EIPs/blob/master/EIPS/eip-721.md

 */

contract ERC721Token is SupportsInterfaceWithLookup, ERC721BasicToken, ERC721 {



  // Token name

  string internal name_;



  // Token symbol

  string internal symbol_;



  // Mapping from owner to list of owned token IDs

  mapping(address => uint256[]) internal ownedTokens;



  // Mapping from token ID to index of the owner tokens list

  mapping(uint256 => uint256) internal ownedTokensIndex;



  // Array with all token ids, used for enumeration

  uint256[] internal allTokens;



  // Mapping from token id to position in the allTokens array

  mapping(uint256 => uint256) internal allTokensIndex;



  // Optional mapping for token URIs

  mapping(uint256 => string) internal tokenURIs;



  /**

   * @dev Constructor function

   */

  constructor(string _name, string _symbol) public {

    name_ = _name;

    symbol_ = _symbol;



    // register the supported interfaces to conform to ERC721 via ERC165

    _registerInterface(InterfaceId_ERC721Enumerable);

    _registerInterface(InterfaceId_ERC721Metadata);

  }



  /**

   * @dev Gets the token name

   * @return string representing the token name

   */

  function name() external view returns (string) {

    return name_;

  }



  /**

   * @dev Gets the token symbol

   * @return string representing the token symbol

   */

  function symbol() external view returns (string) {

    return symbol_;

  }



  /**

   * @dev Returns an URI for a given token ID

   * Throws if the token ID does not exist. May return an empty string.

   * @param _tokenId uint256 ID of the token to query

   */

  function tokenURI(uint256 _tokenId) public view returns (string) {

    require(exists(_tokenId));

    return tokenURIs[_tokenId];

  }



  /**

   * @dev Gets the token ID at a given index of the tokens list of the requested owner

   * @param _owner address owning the tokens list to be accessed

   * @param _index uint256 representing the index to be accessed of the requested tokens list

   * @return uint256 token ID at the given index of the tokens list owned by the requested address

   */

  function tokenOfOwnerByIndex(

    address _owner,

    uint256 _index

  )

    public

    view

    returns (uint256)

  {

    require(_index < balanceOf(_owner));

    return ownedTokens[_owner][_index];

  }



  /**

   * @dev Gets the total amount of tokens stored by the contract

   * @return uint256 representing the total amount of tokens

   */

  function totalSupply() public view returns (uint256) {

    return allTokens.length;

  }



  /**

   * @dev Gets the token ID at a given index of all the tokens in this contract

   * Reverts if the index is greater or equal to the total number of tokens

   * @param _index uint256 representing the index to be accessed of the tokens list

   * @return uint256 token ID at the given index of the tokens list

   */

  function tokenByIndex(uint256 _index) public view returns (uint256) {

    require(_index < totalSupply());

    return allTokens[_index];

  }



  /**

   * @dev Internal function to set the token URI for a given token

   * Reverts if the token ID does not exist

   * @param _tokenId uint256 ID of the token to set its URI

   * @param _uri string URI to assign

   */

  function _setTokenURI(uint256 _tokenId, string _uri) internal {

    require(exists(_tokenId));

    tokenURIs[_tokenId] = _uri;

  }



  /**

   * @dev Internal function to add a token ID to the list of a given address

   * @param _to address representing the new owner of the given token ID

   * @param _tokenId uint256 ID of the token to be added to the tokens list of the given address

   */

  function addTokenTo(address _to, uint256 _tokenId) internal {

    super.addTokenTo(_to, _tokenId);

    uint256 length = ownedTokens[_to].length;

    ownedTokens[_to].push(_tokenId);

    ownedTokensIndex[_tokenId] = length;

  }



  /**

   * @dev Internal function to remove a token ID from the list of a given address

   * @param _from address representing the previous owner of the given token ID

   * @param _tokenId uint256 ID of the token to be removed from the tokens list of the given address

   */

  function removeTokenFrom(address _from, uint256 _tokenId) internal {

    super.removeTokenFrom(_from, _tokenId);



    // To prevent a gap in the array, we store the last token in the index of the token to delete, and

    // then delete the last slot.

    uint256 tokenIndex = ownedTokensIndex[_tokenId];

    uint256 lastTokenIndex = ownedTokens[_from].length.sub(1);

    uint256 lastToken = ownedTokens[_from][lastTokenIndex];



    ownedTokens[_from][tokenIndex] = lastToken;

    // This also deletes the contents at the last position of the array

    ownedTokens[_from].length--;



    // Note that this will handle single-element arrays. In that case, both tokenIndex and lastTokenIndex are going to

    // be zero. Then we can make sure that we will remove _tokenId from the ownedTokens list since we are first swapping

    // the lastToken to the first position, and then dropping the element placed in the last position of the list



    ownedTokensIndex[_tokenId] = 0;

    ownedTokensIndex[lastToken] = tokenIndex;

  }



  /**

   * @dev Internal function to mint a new token

   * Reverts if the given token ID already exists

   * @param _to address the beneficiary that will own the minted token

   * @param _tokenId uint256 ID of the token to be minted by the msg.sender

   */

  function _mint(address _to, uint256 _tokenId) internal {

    super._mint(_to, _tokenId);



    allTokensIndex[_tokenId] = allTokens.length;

    allTokens.push(_tokenId);

  }



  /**

   * @dev Internal function to burn a specific token

   * Reverts if the token does not exist

   * @param _owner owner of the token to burn

   * @param _tokenId uint256 ID of the token being burned by the msg.sender

   */

  function _burn(address _owner, uint256 _tokenId) internal {

    super._burn(_owner, _tokenId);



    // Clear metadata (if any)

    if (bytes(tokenURIs[_tokenId]).length != 0) {

      delete tokenURIs[_tokenId];

    }



    // Reorg all tokens array

    uint256 tokenIndex = allTokensIndex[_tokenId];

    uint256 lastTokenIndex = allTokens.length.sub(1);

    uint256 lastToken = allTokens[lastTokenIndex];



    allTokens[tokenIndex] = lastToken;

    allTokens[lastTokenIndex] = 0;



    allTokens.length--;

    allTokensIndex[_tokenId] = 0;

    allTokensIndex[lastToken] = tokenIndex;

  }



}



// File: @ensdomains/ens/contracts/Deed.sol



/**

 * @title Deed to hold ether in exchange for ownership of a node

 * @dev The deed can be controlled only by the registrar and can only send ether back to the owner.

 */

contract Deed {



    address constant burn = 0xdead;



    address public registrar;

    address public owner;

    address public previousOwner;



    uint public creationDate;

    uint public value;



    bool active;



    event OwnerChanged(address newOwner);

    event DeedClosed();



    modifier onlyRegistrar {

        require(msg.sender == registrar);

        _;

    }



    modifier onlyActive {

        require(active);

        _;

    }



    function Deed(address _owner) public payable {

        owner = _owner;

        registrar = msg.sender;

        creationDate = now;

        active = true;

        value = msg.value;

    }



    function setOwner(address newOwner) public onlyRegistrar {

        require(newOwner != 0);

        previousOwner = owner;  // This allows contracts to check who sent them the ownership

        owner = newOwner;

        OwnerChanged(newOwner);

    }



    function setRegistrar(address newRegistrar) public onlyRegistrar {

        registrar = newRegistrar;

    }



    function setBalance(uint newValue, bool throwOnFailure) public onlyRegistrar onlyActive {

        // Check if it has enough balance to set the value

        require(value >= newValue);

        value = newValue;

        // Send the difference to the owner

        require(owner.send(this.balance - newValue) || !throwOnFailure);

    }



    /**

     * @dev Close a deed and refund a specified fraction of the bid value

     *

     * @param refundRatio The amount*1/1000 to refund

     */

    function closeDeed(uint refundRatio) public onlyRegistrar onlyActive {

        active = false;

        require(burn.send(((1000 - refundRatio) * this.balance)/1000));

        DeedClosed();

        destroyDeed();

    }



    /**

     * @dev Close a deed and refund a specified fraction of the bid value

     */

    function destroyDeed() public {

        require(!active);



        // Instead of selfdestruct(owner), invoke owner fallback function to allow

        // owner to log an event if desired; but owner should also be aware that

        // its fallback function can also be invoked by setBalance

        if (owner.send(this.balance)) {

            selfdestruct(burn);

        }

    }

}



// File: @ensdomains/ens/contracts/ENS.sol



interface ENS {



    // Logged when the owner of a node assigns a new owner to a subnode.

    event NewOwner(bytes32 indexed node, bytes32 indexed label, address owner);



    // Logged when the owner of a node transfers ownership to a new account.

    event Transfer(bytes32 indexed node, address owner);



    // Logged when the resolver for a node changes.

    event NewResolver(bytes32 indexed node, address resolver);



    // Logged when the TTL of a node changes

    event NewTTL(bytes32 indexed node, uint64 ttl);





    function setSubnodeOwner(bytes32 node, bytes32 label, address owner) public;

    function setResolver(bytes32 node, address resolver) public;

    function setOwner(bytes32 node, address owner) public;

    function setTTL(bytes32 node, uint64 ttl) public;

    function owner(bytes32 node) public view returns (address);

    function resolver(bytes32 node) public view returns (address);

    function ttl(bytes32 node) public view returns (uint64);



}



// File: @ensdomains/ens/contracts/HashRegistrarSimplified.sol



/*



Temporary Hash Registrar

========================



This is a simplified version of a hash registrar. It is purporsefully limited:

names cannot be six letters or shorter, new auctions will stop after 4 years.



The plan is to test the basic features and then move to a new contract in at most

2 years, when some sort of renewal mechanism will be enabled.

*/







/**

 * @title Registrar

 * @dev The registrar handles the auction process for each subnode of the node it owns.

 */

contract Registrar {

    ENS public ens;

    bytes32 public rootNode;



    mapping (bytes32 => Entry) _entries;

    mapping (address => mapping (bytes32 => Deed)) public sealedBids;

    

    enum Mode { Open, Auction, Owned, Forbidden, Reveal, NotYetAvailable }



    uint32 constant totalAuctionLength = 5 days;

    uint32 constant revealPeriod = 48 hours;

    uint32 public constant launchLength = 8 weeks;



    uint constant minPrice = 0.01 ether;

    uint public registryStarted;



    event AuctionStarted(bytes32 indexed hash, uint registrationDate);

    event NewBid(bytes32 indexed hash, address indexed bidder, uint deposit);

    event BidRevealed(bytes32 indexed hash, address indexed owner, uint value, uint8 status);

    event HashRegistered(bytes32 indexed hash, address indexed owner, uint value, uint registrationDate);

    event HashReleased(bytes32 indexed hash, uint value);

    event HashInvalidated(bytes32 indexed hash, string indexed name, uint value, uint registrationDate);



    struct Entry {

        Deed deed;

        uint registrationDate;

        uint value;

        uint highestBid;

    }



    modifier inState(bytes32 _hash, Mode _state) {

        require(state(_hash) == _state);

        _;

    }



    modifier onlyOwner(bytes32 _hash) {

        require(state(_hash) == Mode.Owned && msg.sender == _entries[_hash].deed.owner());

        _;

    }



    modifier registryOpen() {

        require(now >= registryStarted && now <= registryStarted + 4 years && ens.owner(rootNode) == address(this));

        _;

    }



    /**

     * @dev Constructs a new Registrar, with the provided address as the owner of the root node.

     *

     * @param _ens The address of the ENS

     * @param _rootNode The hash of the rootnode.

     */

    function Registrar(ENS _ens, bytes32 _rootNode, uint _startDate) public {

        ens = _ens;

        rootNode = _rootNode;

        registryStarted = _startDate > 0 ? _startDate : now;

    }



    /**

     * @dev Start an auction for an available hash

     *

     * @param _hash The hash to start an auction on

     */

    function startAuction(bytes32 _hash) public registryOpen() {

        Mode mode = state(_hash);

        if (mode == Mode.Auction) return;

        require(mode == Mode.Open);



        Entry storage newAuction = _entries[_hash];

        newAuction.registrationDate = now + totalAuctionLength;

        newAuction.value = 0;

        newAuction.highestBid = 0;

        AuctionStarted(_hash, newAuction.registrationDate);

    }



    /**

     * @dev Start multiple auctions for better anonymity

     *

     * Anyone can start an auction by sending an array of hashes that they want to bid for.

     * Arrays are sent so that someone can open up an auction for X dummy hashes when they

     * are only really interested in bidding for one. This will increase the cost for an

     * attacker to simply bid blindly on all new auctions. Dummy auctions that are

     * open but not bid on are closed after a week.

     *

     * @param _hashes An array of hashes, at least one of which you presumably want to bid on

     */

    function startAuctions(bytes32[] _hashes) public {

        for (uint i = 0; i < _hashes.length; i ++) {

            startAuction(_hashes[i]);

        }

    }



    /**

     * @dev Submit a new sealed bid on a desired hash in a blind auction

     *

     * Bids are sent by sending a message to the main contract with a hash and an amount. The hash

     * contains information about the bid, including the bidded hash, the bid amount, and a random

     * salt. Bids are not tied to any one auction until they are revealed. The value of the bid

     * itself can be masqueraded by sending more than the value of your actual bid. This is

     * followed by a 48h reveal period. Bids revealed after this period will be burned and the ether unrecoverable.

     * Since this is an auction, it is expected that most public hashes, like known domains and common dictionary

     * words, will have multiple bidders pushing the price up.

     *

     * @param sealedBid A sealedBid, created by the shaBid function

     */

    function newBid(bytes32 sealedBid) public payable {

        require(address(sealedBids[msg.sender][sealedBid]) == 0x0);

        require(msg.value >= minPrice);



        // Creates a new hash contract with the owner

        Deed newBid = (new Deed).value(msg.value)(msg.sender);

        sealedBids[msg.sender][sealedBid] = newBid;

        NewBid(sealedBid, msg.sender, msg.value);

    }



    /**

     * @dev Start a set of auctions and bid on one of them

     *

     * This method functions identically to calling `startAuctions` followed by `newBid`,

     * but all in one transaction.

     *

     * @param hashes A list of hashes to start auctions on.

     * @param sealedBid A sealed bid for one of the auctions.

     */

    function startAuctionsAndBid(bytes32[] hashes, bytes32 sealedBid) public payable {

        startAuctions(hashes);

        newBid(sealedBid);

    }



    /**

     * @dev Submit the properties of a bid to reveal them

     *

     * @param _hash The node in the sealedBid

     * @param _value The bid amount in the sealedBid

     * @param _salt The sale in the sealedBid

     */

    function unsealBid(bytes32 _hash, uint _value, bytes32 _salt) public {

        bytes32 seal = shaBid(_hash, msg.sender, _value, _salt);

        Deed bid = sealedBids[msg.sender][seal];

        require(address(bid) != 0);



        sealedBids[msg.sender][seal] = Deed(0);

        Entry storage h = _entries[_hash];

        uint value = min(_value, bid.value());

        bid.setBalance(value, true);



        var auctionState = state(_hash);

        if (auctionState == Mode.Owned) {

            // Too late! Bidder loses their bid. Gets 0.5% back.

            bid.closeDeed(5);

            BidRevealed(_hash, msg.sender, value, 1);

        } else if (auctionState != Mode.Reveal) {

            // Invalid phase

            revert();

        } else if (value < minPrice || bid.creationDate() > h.registrationDate - revealPeriod) {

            // Bid too low or too late, refund 99.5%

            bid.closeDeed(995);

            BidRevealed(_hash, msg.sender, value, 0);

        } else if (value > h.highestBid) {

            // New winner

            // Cancel the other bid, refund 99.5%

            if (address(h.deed) != 0) {

                Deed previousWinner = h.deed;

                previousWinner.closeDeed(995);

            }



            // Set new winner

            // Per the rules of a vickery auction, the value becomes the previous highestBid

            h.value = h.highestBid;  // will be zero if there's only 1 bidder

            h.highestBid = value;

            h.deed = bid;

            BidRevealed(_hash, msg.sender, value, 2);

        } else if (value > h.value) {

            // Not winner, but affects second place

            h.value = value;

            bid.closeDeed(995);

            BidRevealed(_hash, msg.sender, value, 3);

        } else {

            // Bid doesn't affect auction

            bid.closeDeed(995);

            BidRevealed(_hash, msg.sender, value, 4);

        }

    }



    /**

     * @dev Cancel a bid

     *

     * @param seal The value returned by the shaBid function

     */

    function cancelBid(address bidder, bytes32 seal) public {

        Deed bid = sealedBids[bidder][seal];

        

        // If a sole bidder does not `unsealBid` in time, they have a few more days

        // where they can call `startAuction` (again) and then `unsealBid` during

        // the revealPeriod to get back their bid value.

        // For simplicity, they should call `startAuction` within

        // 9 days (2 weeks - totalAuctionLength), otherwise their bid will be

        // cancellable by anyone.

        require(address(bid) != 0 && now >= bid.creationDate() + totalAuctionLength + 2 weeks);



        // Send the canceller 0.5% of the bid, and burn the rest.

        bid.setOwner(msg.sender);

        bid.closeDeed(5);

        sealedBids[bidder][seal] = Deed(0);

        BidRevealed(seal, bidder, 0, 5);

    }



    /**

     * @dev Finalize an auction after the registration date has passed

     *

     * @param _hash The hash of the name the auction is for

     */

    function finalizeAuction(bytes32 _hash) public onlyOwner(_hash) {

        Entry storage h = _entries[_hash];

        

        // Handles the case when there's only a single bidder (h.value is zero)

        h.value =  max(h.value, minPrice);

        h.deed.setBalance(h.value, true);



        trySetSubnodeOwner(_hash, h.deed.owner());

        HashRegistered(_hash, h.deed.owner(), h.value, h.registrationDate);

    }



    /**

     * @dev The owner of a domain may transfer it to someone else at any time.

     *

     * @param _hash The node to transfer

     * @param newOwner The address to transfer ownership to

     */

    function transfer(bytes32 _hash, address newOwner) public onlyOwner(_hash) {

        require(newOwner != 0);



        Entry storage h = _entries[_hash];

        h.deed.setOwner(newOwner);

        trySetSubnodeOwner(_hash, newOwner);

    }



    /**

     * @dev After some time, or if we're no longer the registrar, the owner can release

     *      the name and get their ether back.

     *

     * @param _hash The node to release

     */

    function releaseDeed(bytes32 _hash) public onlyOwner(_hash) {

        Entry storage h = _entries[_hash];

        Deed deedContract = h.deed;



        require(now >= h.registrationDate + 1 years || ens.owner(rootNode) != address(this));



        h.value = 0;

        h.highestBid = 0;

        h.deed = Deed(0);



        _tryEraseSingleNode(_hash);

        deedContract.closeDeed(1000);

        HashReleased(_hash, h.value);        

    }



    /**

     * @dev Submit a name 6 characters long or less. If it has been registered,

     *      the submitter will earn 50% of the deed value. 

     * 

     * We are purposefully handicapping the simplified registrar as a way 

     * to force it into being restructured in a few years.

     *

     * @param unhashedName An invalid name to search for in the registry.

     */

    function invalidateName(string unhashedName) public inState(keccak256(unhashedName), Mode.Owned) {

        require(strlen(unhashedName) <= 6);

        bytes32 hash = keccak256(unhashedName);



        Entry storage h = _entries[hash];



        _tryEraseSingleNode(hash);



        if (address(h.deed) != 0) {

            // Reward the discoverer with 50% of the deed

            // The previous owner gets 50%

            h.value = max(h.value, minPrice);

            h.deed.setBalance(h.value/2, false);

            h.deed.setOwner(msg.sender);

            h.deed.closeDeed(1000);

        }



        HashInvalidated(hash, unhashedName, h.value, h.registrationDate);



        h.value = 0;

        h.highestBid = 0;

        h.deed = Deed(0);

    }



    /**

     * @dev Allows anyone to delete the owner and resolver records for a (subdomain of) a

     *      name that is not currently owned in the registrar. If passing, eg, 'foo.bar.eth',

     *      the owner and resolver fields on 'foo.bar.eth' and 'bar.eth' will all be cleared.

     *

     * @param labels A series of label hashes identifying the name to zero out, rooted at the

     *        registrar's root. Must contain at least one element. For instance, to zero 

     *        'foo.bar.eth' on a registrar that owns '.eth', pass an array containing

     *        [keccak256('foo'), keccak256('bar')].

     */

    function eraseNode(bytes32[] labels) public {

        require(labels.length != 0);

        require(state(labels[labels.length - 1]) != Mode.Owned);



        _eraseNodeHierarchy(labels.length - 1, labels, rootNode);

    }



    /**

     * @dev Transfers the deed to the current registrar, if different from this one.

     *

     * Used during the upgrade process to a permanent registrar.

     *

     * @param _hash The name hash to transfer.

     */

    function transferRegistrars(bytes32 _hash) public onlyOwner(_hash) {

        address registrar = ens.owner(rootNode);

        require(registrar != address(this));



        // Migrate the deed

        Entry storage h = _entries[_hash];

        h.deed.setRegistrar(registrar);



        // Call the new registrar to accept the transfer

        Registrar(registrar).acceptRegistrarTransfer(_hash, h.deed, h.registrationDate);



        // Zero out the Entry

        h.deed = Deed(0);

        h.registrationDate = 0;

        h.value = 0;

        h.highestBid = 0;

    }



    /**

     * @dev Accepts a transfer from a previous registrar; stubbed out here since there

     *      is no previous registrar implementing this interface.

     *

     * @param hash The sha3 hash of the label to transfer.

     * @param deed The Deed object for the name being transferred in.

     * @param registrationDate The date at which the name was originally registered.

     */

    function acceptRegistrarTransfer(bytes32 hash, Deed deed, uint registrationDate) public {

        hash; deed; registrationDate; // Don't warn about unused variables

    }



    // State transitions for names:

    //   Open -> Auction (startAuction)

    //   Auction -> Reveal

    //   Reveal -> Owned

    //   Reveal -> Open (if nobody bid)

    //   Owned -> Open (releaseDeed or invalidateName)

    function state(bytes32 _hash) public view returns (Mode) {

        Entry storage entry = _entries[_hash];



        if (!isAllowed(_hash, now)) {

            return Mode.NotYetAvailable;

        } else if (now < entry.registrationDate) {

            if (now < entry.registrationDate - revealPeriod) {

                return Mode.Auction;

            } else {

                return Mode.Reveal;

            }

        } else {

            if (entry.highestBid == 0) {

                return Mode.Open;

            } else {

                return Mode.Owned;

            }

        }

    }



    function entries(bytes32 _hash) public view returns (Mode, address, uint, uint, uint) {

        Entry storage h = _entries[_hash];

        return (state(_hash), h.deed, h.registrationDate, h.value, h.highestBid);

    }



    /**

     * @dev Determines if a name is available for registration yet

     *

     * Each name will be assigned a random date in which its auction

     * can be started, from 0 to 8 weeks

     *

     * @param _hash The hash to start an auction on

     * @param _timestamp The timestamp to query about

     */

    function isAllowed(bytes32 _hash, uint _timestamp) public view returns (bool allowed) {

        return _timestamp > getAllowedTime(_hash);

    }



    /**

     * @dev Returns available date for hash

     *

     * The available time from the `registryStarted` for a hash is proportional

     * to its numeric value.

     *

     * @param _hash The hash to start an auction on

     */

    function getAllowedTime(bytes32 _hash) public view returns (uint) {

        return registryStarted + ((launchLength * (uint(_hash) >> 128)) >> 128);

        // Right shift operator: a >> b == a / 2**b

    }



    /**

     * @dev Hash the values required for a secret bid

     *

     * @param hash The node corresponding to the desired namehash

     * @param value The bid amount

     * @param salt A random value to ensure secrecy of the bid

     * @return The hash of the bid values

     */

    function shaBid(bytes32 hash, address owner, uint value, bytes32 salt) public pure returns (bytes32) {

        return keccak256(hash, owner, value, salt);

    }



    function _tryEraseSingleNode(bytes32 label) internal {

        if (ens.owner(rootNode) == address(this)) {

            ens.setSubnodeOwner(rootNode, label, address(this));

            bytes32 node = keccak256(rootNode, label);

            ens.setResolver(node, 0);

            ens.setOwner(node, 0);

        }

    }



    function _eraseNodeHierarchy(uint idx, bytes32[] labels, bytes32 node) internal {

        // Take ownership of the node

        ens.setSubnodeOwner(node, labels[idx], address(this));

        node = keccak256(node, labels[idx]);



        // Recurse if there are more labels

        if (idx > 0) {

            _eraseNodeHierarchy(idx - 1, labels, node);

        }



        // Erase the resolver and owner records

        ens.setResolver(node, 0);

        ens.setOwner(node, 0);

    }



    /**

     * @dev Assign the owner in ENS, if we're still the registrar

     *

     * @param _hash hash to change owner

     * @param _newOwner new owner to transfer to

     */

    function trySetSubnodeOwner(bytes32 _hash, address _newOwner) internal {

        if (ens.owner(rootNode) == address(this))

            ens.setSubnodeOwner(rootNode, _hash, _newOwner);

    }



    /**

     * @dev Returns the maximum of two unsigned integers

     *

     * @param a A number to compare

     * @param b A number to compare

     * @return The maximum of two unsigned integers

     */

    function max(uint a, uint b) internal pure returns (uint) {

        if (a > b)

            return a;

        else

            return b;

    }



    /**

     * @dev Returns the minimum of two unsigned integers

     *

     * @param a A number to compare

     * @param b A number to compare

     * @return The minimum of two unsigned integers

     */

    function min(uint a, uint b) internal pure returns (uint) {

        if (a < b)

            return a;

        else

            return b;

    }



    /**

     * @dev Returns the length of a given string

     *

     * @param s The string to measure the length of

     * @return The length of the input string

     */

    function strlen(string s) internal pure returns (uint) {

        s; // Don't warn about unused variables

        // Starting here means the LSB will be the byte we care about

        uint ptr;

        uint end;

        assembly {

            ptr := add(s, 1)

            end := add(mload(s), ptr)

        }

        for (uint len = 0; ptr < end; len++) {

            uint8 b;

            assembly { b := and(mload(ptr), 0xFF) }

            if (b < 0x80) {

                ptr += 1;

            } else if (b < 0xE0) {

                ptr += 2;

            } else if (b < 0xF0) {

                ptr += 3;

            } else if (b < 0xF8) {

                ptr += 4;

            } else if (b < 0xFC) {

                ptr += 5;

            } else {

                ptr += 6;

            }

        }

        return len;

    }



}



// File: contracts/ENSNFT.sol



contract ENSNFT is ERC721Token {

    event Transfer(address indexed _from, address indexed _to, uint256 indexed _tokenId);

    Registrar registrar;

    constructor (string _name, string _symbol, address _registrar) public

        ERC721Token(_name, _symbol) {

        registrar = Registrar(_registrar);

    }

    function mint(bytes32 _hash) public {

        address deedAddress;

        (, deedAddress, , , ) = registrar.entries(_hash);

        Deed deed = Deed(deedAddress);

        require(deed.owner() == address(this));

        require(deed.previousOwner() == msg.sender);

        uint256 tokenId = uint256(_hash); // dont do math on this

        _mint(deed.previousOwner(), tokenId);

    }

    function burn(uint256 tokenId) {

        require(ownerOf(tokenId) == msg.sender);

        _burn(msg.sender, tokenId);

        registrar.transfer(bytes32(tokenId), msg.sender);

    }

}