%------------------------------------------------------------------------------
%       boxpage class
%       =============
%
%       Provides a picture environment to position objects on slides.
%
%       boxpage is based on, and uses, the slides class. A new 
%       environment "boxpage" is defined which basically provides 
%       a full-page picture environment in which boxes can be
%       positioned at coordinates defined by a user-settable scale
%       (default=0.1mm). The central command is \putbox, which offers
%       a large variety of possibilities to define boxes (including
%       ones containing postscript figures) with different user-settable
%       properties (size, colors, optional frames, position adjustment).
%       Page decorations to be repeated on each page can also be defined. 
%
%       More details in the following comments. 
%
%       Author: U. Katz, University of Bonn, April 2000
%               Bug reports etc. to katz@physik.uni-bonn.de
%
%------------------------------------------------------------------------------
%------------------------------------------------------------------------------
%       Class name
%       ==========
%------------------------------------------------------------------------------
\ProvidesClass{./LaTeX/styles/boxpage}[2000/04/15 Version 0.9]
\NeedsTeXFormat{LaTeX2e}
%------------------------------------------------------------------------------
%       Declare and process options
%       ===========================
%       [showbox]                -> draw box around each page (for debugging) 
%       [noshowbox]              -> don't draw box (default)
%       [showgrid]               -> draw grid on each page (for development)
%       [noshowgrid]             -> don't draw grid (default)
%
%       all other options are passed to the slides package
%
%       User-settable logicals:
%       -----------------------
%       ifboxpageframe (false)   -> draw box around each page (for debugging) 
%       ifboxpagegrid (false)    -> draw grid on each page (for development) 
%------------------------------------------------------------------------------
\newif\ifboxpageframe
\newif\ifboxpagegrid
\DeclareOption{showbox}{\boxpageframetrue}
\DeclareOption{noshowbox}{\boxpageframefalse}
\DeclareOption{showgrid}{\boxpagegridtrue}
\DeclareOption{noshowgrid}{\boxpagegridfalse}
\ExecuteOptions{noshowbox,noshowgrid}
\DeclareOption*{\PassOptionsToClass{\CurrentOption}{slides}}
\ProcessOptions\relax
%------------------------------------------------------------------------------
%       Load the slides class and pass options
%       ======================================
%------------------------------------------------------------------------------
\LoadClass{slides}
%------------------------------------------------------------------------------
%       Dependencies on other packages
%       ==============================
%------------------------------------------------------------------------------
\RequirePackage{calc}
\RequirePackage{ifthen}
\RequirePackage{color}
\RequirePackage{epsfig}
\RequirePackage{rotating}
\RequirePackage{epic}
%------------------------------------------------------------------------------
%       Internal definitions
%       ====================
%------------------------------------------------------------------------------
\catcode`\@=11 % @ signs are now treated as letters
%------------------------------------------------------------------------------
%       Internal length registers
%       -------------------------
%       boxpict@scale            -> unit length (0.1mm)
%       divlen@divisor           -> divide lengths: divisor length
%       divlen@dividend          -> divide lengths: dividend length
%       pb@width                 -> total width of putbox
%       pb@height                -> height of putbox
%       pb@depth                 -> depth of putbox
%       pb@toth                  -> height+depth of putbox
%       pb@mpwidth               -> width of minipage inside putbox
%       pb@mptoth                -> height+depth of minipage inside putbox
%       pb@frwidth               -> \fboxrule+\fboxsep
%       pb@rboffset              -> offset to be added to \pb@rbheigh
%       pb@rbheight              -> raisebox height for vertical adjustment
%------------------------------------------------------------------------------
\newlength{\boxpict@scale}
\newlength{\divlen@divisor}
\newlength{\divlen@dividend}
\newlength{\pb@width}
\newlength{\pb@height}
\newlength{\pb@depth}
\newlength{\pb@toth}
\newlength{\pb@mpwidth}
\newlength{\pb@mptoth}
\newlength{\pb@frwidth}
\newlength{\pb@rboffset}
\newlength{\pb@rbheight}
%------------------------------------------------------------------------------
%       Internal counters
%       -----------------
%       boxpage@width            -> text width in units of boxpict@scale 
%       boxpage@height           -> text height in units of boxpict@scale 
%       pb@bpsw                  -> width of putbox in units of \boxpict@scale
%       pb@bpsx                  -> x position in units of \boxpict@scale
%       pb@bpsy                  -> y position in units of \boxpict@scale
%       pb@bpfr                  -> horizontal offset in units of \boxpict@scale
%       divlen@rata              -> divide lengths: counter for integer part
%       divlen@ratb              -> divide lengths: counter for fractional part
%       divlen@ratt              -> divide lengths: auxiliary counter
%       divlen@cnt               -> divide lengths: loop counter
%------------------------------------------------------------------------------
\newcounter{boxpage@width}
\newcounter{boxpage@height}
\newcounter{pb@bpsw}
\newcounter{pb@bpsx}
\newcounter{pb@bpsy}
\newcounter{pb@bpfr}
\newcounter{divlen@rata}
\newcounter{divlen@ratb}
\newcounter{divlen@ratt}
\newcounter{divlen@cnt}
%------------------------------------------------------------------------------
%       Internal box registers
%       ----------------------
%       pb@box                   -> box to be written out
%       bp@deco                  -> box to save bo page decoration
%------------------------------------------------------------------------------
\newsavebox{\pb@box}
\newsavebox{\bp@deco}
%------------------------------------------------------------------------------
%       Internal auxiliary commands
%       ---------------------------
%       boxpage@arg              -> to store argument of boxpge environment
%       putbox@cfra              -> frame color of boxes
%       putbox@cbkg              -> background color of boxes
%       putbox@ctxt              -> text color of boxes
%       putbox@fsep              -> frame separation for framed boxes
%                                   (used to save value of \fboxsep globally)
%       putbox@fwid              -> frame width for framed boxes
%                                   (used to save value of \fboxrule globally)
%       putbox@cfra@save         -> 
%       putbox@cbkg@save         -> 
%       putbox@ctxt@save         -> like above parameters, saved by \putboxsave
%       putbox@fsep@save         -> 
%       putbox@fsep@save         -> 
%       putbox@scal@save         -> boxpict@scale parameter saved by \putboxsave
%       boxpage@pictscale        -> scale factor in picture environment
%       boxpage@pictwidth        -> line width in picture environment
%       divlen@res               -> divide lengths: result
%------------------------------------------------------------------------------
\newcommand{\boxpage@arg}{}
\newcommand{\putbox@cfra}{}
\newcommand{\putbox@cbkg}{}
\newcommand{\putbox@ctxt}{}
\newcommand{\putbox@fsep}{3.pt}
\newcommand{\putbox@fwid}{0.4pt}
\newcommand{\putbox@cfra@save}{black}
\newcommand{\putbox@cbkg@save}{white}
\newcommand{\putbox@ctxt@save}{\black}
\newcommand{\putbox@fsep@save}{3.pt}
\newcommand{\putbox@fwid@save}{0.4pt}
\newcommand{\putbox@scal@save}{0.1mm}
\newcommand{\boxpage@pictscale}{}
\newcommand{\boxpage@pictwidth}{}
\newcommand{\divlen@res}{}
%------------------------------------------------------------------------------
%       Internal boolean flags
%       ----------------------
%       ifpb@mpheight            -> true if box height is explicitly given
%------------------------------------------------------------------------------
\newif\ifpb@mpheight
%------------------------------------------------------------------------------
%       External definitions
%       ====================
%------------------------------------------------------------------------------
%------------------------------------------------------------------------------
%       The boxpage environment
%       -----------------------
%       argument = n    normal (default)
%                  e    empty (no \boxpagedecoration)
%                  o    like e, but suppresses page counter advance
%                       (useful for overlays)
%                  p    plain (usual slides environment,
%                              \boxpagedecoration must be added by hand)
%
%       User-redefinable commands:
%       -------------------------
%       bpwidth                  -> contents of boxpage@width
%                                   (textwidth in units of \boxpict@scale)
%       bpheight                 -> contents of boxpage@height
%       gridmajor                -> numbered grid spacing (default=200) 
%                                   (in units of \boxpict@scale)
%       gridminor                -> unnumbered subgrid spacing (default=50)
%                                   (in units of \boxpict@scale)
%       gridmajorcol             -> major grid color (default=midgray)
%       gridminorcol             -> minor grid color (default=lightergray)
%------------------------------------------------------------------------------
\newcommand{\bpwidth}{}
\newcommand{\bpheight}{}
\newcommand{\gridmajor}{200}
\newcommand{\gridminor}{50}
\newcommand{\gridmajorcol}{midgray}
\newcommand{\gridminorcol}{lightergray}
\newenvironment{boxpage}[1][n]%
 {\ifthenelse{\equal{#1}{o}}{\setcounter{slide}{\value{slide}-1}}{}%
  \ifthenelse{\equal{#1}{e}\or\equal{#1}{o}\or\equal{#1}{n}\or\equal{#1}{p}}{}%
     {\PackageError{boxpage}{^^JInvalid argument of boxpage%
                            (must be one of n, e, o, p)}}
  \begin{slide}
  \renewcommand{\boxpage@arg}{#1}
  \ifthenelse{\equal{#1}{p}}{}{%
    \setlength{\unitlength}{\boxpict@scale}
    \setcounter{boxpage@width}%
               {1*\ratio{\textwidth+\boxpict@scale/2}{\boxpict@scale}}
    \setcounter{boxpage@height}%
               {1*\ratio{\textheight+\boxpict@scale/2}{\boxpict@scale}}
    \renewcommand{\bpwidth}{\theboxpage@width}
    \renewcommand{\bpheight}{\theboxpage@height}
    \dividelengths{\boxpict@scale}{1.bp}{5}
    \global\edef\boxpage@pictscale{\divlen@res}
    \dividelengths{0.5bp}{\boxpict@scale}{5}
    \global\edef\boxpage@pictwidth{\divlen@res}
    \begin{picture}(\theboxpage@width,\theboxpage@height)
    \SetScale{\boxpage@pictscale}
    \SetWidth{\boxpage@pictwidth}
    \ifboxpagegrid{\put(0,0){\color{\gridminorcol}%
                   \bpgrid(\bpwidth,\bpheight)(\gridminor,\gridminor)}%
                   \put(0,0){\color{\gridmajorcol}\tiny%
                   \bpgrid(\bpwidth,\bpheight)(\gridmajor,\gridmajor)[0,0]}%
                  }\fi
    \ifthenelse{\equal{#1}{e}\or\equal{#1}{o}}{}{\putboxpagedecoration}}
 }%
 {\par
  \ifthenelse{\equal{\boxpage@arg}{p}}{}{%
    \ifboxpageframe{\EBox(0,0)(\theboxpage@width,\theboxpage@height)}\fi
    \end{picture}}
  \end{slide}}
%------------------------------------------------------------------------------
%       \putboxpar and related commands (parameter initialization)
%       ----------------------------------------------------------
%       define colors, framewidth etc. to be used in \putbox commands
%
%       \putboxpar[scale]{fdist}{fwdth}{cfra}{cbkg}{ctxt}
%
%               scale = sets \boxpict@scale (default = 0.1mm)
%               fdist = distance of frame from box contents [3.pt]
%               fwdth = frame linewidth [0.4pt]
%               cfra  = color of frame [black]
%               cbkg  = color of background [white]
%               ctxt  = color of text [\black]
%
%               Empty arguments are not changed.
%               The initialization presets values in square brackets.
%
%       \putboxparloc[scale]{fdist}{fwdth}{cfra}{cbkg}{ctxt}
%
%               like \putboxpar, but resets parameters only locally
%
%       \putboxcol{cfra}{cbkg}{ctxt}
%
%               cfra  = color of frame [black]
%               cbkg  = color of background [white]
%               ctxt  = color of text [\black]
%
%               Empty arguments are not changed.
%               The initialization presets values in square brackets.
%
%       \putboxcolloc{cfra}{cbkg}{ctxt}
%
%               like \putboxcol, but resets parameters only locally
%
%       \putboxfra{fdist}{fwdth}
%
%               fdist = distance of frame from box contents [3.pt]
%               fwdth = frame linewidth [0.4pt]
%
%               Empty arguments are not changed.
%               The initialization presets values in square brackets.
%
%       \putboxfraloc{fdist}{fwdth}
%
%               like \putboxfra, but resets parameters only locally
%
%       \putboxsca{scale}
%
%               scale = sets \boxpict@scale (default = 0.1mm)
%
%               The initialization presets the value in square brackets.
%
%       \putboxscaloc{scale}
%
%               like \putboxsca, but resets the scale only locally
%
%       \putboxdefault
%
%               resets all parameters to defaults
%------------------------------------------------------------------------------
\newcommand{\putboxpar}[6][0.1mm]{%
  \setlength{\boxpict@scale}{#1}
  \ifthenelse{\equal{#2}{}}{}{\global\def\putbox@fsep{#2}}
  \ifthenelse{\equal{#3}{}}{}{\global\def\putbox@fwid{#3}}
  \ifthenelse{\equal{#4}{}}{}{\global\def\putbox@cfra{#4}}
  \ifthenelse{\equal{#5}{}}{}{\global\def\putbox@cbkg{#5}}
  \ifthenelse{\equal{#6}{}}{}{\global\def\putbox@ctxt{#6}}
}
\newcommand{\putboxparloc}[6][0.1mm]{%
  \setlength{\boxpict@scale}{#1}
  \ifthenelse{\equal{#2}{}}{}{\renewcommand{\putbox@fsep}{#2}}
  \ifthenelse{\equal{#3}{}}{}{\renewcommand{\putbox@fwid}{#3}}
  \ifthenelse{\equal{#4}{}}{}{\renewcommand{\putbox@cfra}{#4}}
  \ifthenelse{\equal{#5}{}}{}{\renewcommand{\putbox@cbkg}{#5}}
  \ifthenelse{\equal{#6}{}}{}{\renewcommand{\putbox@ctxt}{#6}}
}
\newcommand{\putboxcol}[3]{%
  \ifthenelse{\equal{#1}{}}{}{\global\def\putbox@cfra{#1}}
  \ifthenelse{\equal{#2}{}}{}{\global\def\putbox@cbkg{#2}}
  \ifthenelse{\equal{#3}{}}{}{\global\def\putbox@ctxt{#3}}
}
\newcommand{\putboxcolloc}[3]{%
  \ifthenelse{\equal{#1}{}}{}{\renewcommand{\putbox@cfra}{#1}}
  \ifthenelse{\equal{#2}{}}{}{\renewcommand{\putbox@cbkg}{#2}}
  \ifthenelse{\equal{#3}{}}{}{\renewcommand{\putbox@ctxt}{#3}}
}
\newcommand{\putboxfra}[2]{%
  \ifthenelse{\equal{#1}{}}{}{\global\def\putbox@fsep{#1}}
  \ifthenelse{\equal{#2}{}}{}{\global\def\putbox@fwid{#2}}
}
\newcommand{\putboxfraloc}[2]{%
  \ifthenelse{\equal{#1}{}}{}{\renewcommand{\putbox@fsep}{#1}}
  \ifthenelse{\equal{#2}{}}{}{\renewcommand{\putbox@fwid}{#2}}
}
\newcommand{\putboxsca}[1]{%
  \setlength{\boxpict@scale}{#1}
}
\newcommand{\putboxscaloc}[1]{%
  \setlength{\boxpict@scale}{#1}
}
\newcommand{\putboxdefault}{%
  \putboxpar[0.1mm]{3.pt}{0.4pt}{black}{white}{black}
}
%------------------------------------------------------------------------------
%       The all-purpose \putbox command
%       -------------------------------
%       Colors, framewidth etc. can be adjusted by \putboxpar et al.
%       All lengths are in units of \boxpict@scale.
%
%       \putbox[angle]{type}{posb}{posi}{xpos}{ypos}{wdth}{hght}{cont}
%                 1      2     3     4     5     6     7     8     9
%
%               angle = rotation angle around box reference point 
%                       (in degrees, counterclockwise, default=0)
%               type  = b   plain box, providing minipage environment
%                       B   like b, but with additional frame distance
%                           around it,
%                       F   framed plain box, providing minipage environment
%                       c   box with colored background (\colorbox), 
%                           providing minipage environment
%                       f   framed box with colored background (\fcolorbox),
%                           providing minipage environment
%                       p   encapsulated postscript file
%               posb  = two-character flag 'hv' indicating position of reference
%                       point w.r.t. box (also determines center of rotation ...
%                       caution: rotation is always around left edge ...)
%                       h = l   left edge
%                       h = c   horizontally centered
%                       h = r   right edge
%                       v = t   top edge
%                       v = c   vertically centered
%                       v = b   bottom edge
%               posi  = one-character flag 'v' indicating vertical position of
%                       text in box (used as argument of minipage environment)
%                       v = t   top 
%                       v = c   centered
%                       v = b   bottom
%                       v = s   spread in over vertical dimension, requires 
%                               vertical rubber lengths (\vfill etc.)
%                       ignored if type=p or hgth=0
%               xpos  = x position of box reference point w.r.t boxpage 
%                       picture, in units of \boxpict@scale;
%                       positive number or  '0': distance from left margin
%                       negative number or '-0': distance from right margin
%                       'c'                    : horizontally centered
%               ypos  = y position of box reference point w.r.t boxpage 
%                       picture, in units of \boxpict@scale;
%                       positive number or  '0': distance from top margin
%                       negative number or '-0': distance from bottom margin
%                       'c'                    : vertically centered
%               wdth  = box width in units of \boxpict@scale, 
%                       = 0 for natural width (caution: inadequate for boxes
%                           containig "vertical structure", e.g. more than
%                           one line of text or a center environment etc.)
%                       = s for \textwidth
%               hght  = box height in units of \boxpict@scale, 
%                       = 0 for natural height 
%                       = s for \textheight
%               cont  = box contents. If type=p, cont defines argument of
%                       \psfig, where the width and/or height tags are added
%                       internally; minimum requirement: figure=<file name>.
%
%               xpos, ypos, wdth and hght can be given using \bpwidth and 
%               \bpheight. Arithmic expressions involving integers are also
%               allowed, real numbers have to be given as \real{<number>}. To
%               produce a box of half of the page width, for example, one can
%               specify wdth as "\bpwidth/2" or as "\bpwidth*50/100" or
%               as "\bpwidth*\real{0.5}".
%------------------------------------------------------------------------------
\newcommand{\putbox}[9][0]{%
  \ifthenelse{\equal{#2}{b}\or\equal{#2}{B}\or\equal{#2}{f}\or\equal{#2}{F}\or%
              \equal{#2}{c}\or\equal{#2}{p}}{}%
     {\PackageError{boxpage}{^^JInvalid argument "type" for putbox %
                            (must be one of b, B, F, c, F, p)}}
  \ifthenelse{\equal{#3}{lt}\or\equal{#3}{lc}\or\equal{#3}{lb}\or%
              \equal{#3}{ct}\or\equal{#3}{cc}\or\equal{#3}{cb}\or%
              \equal{#3}{rt}\or\equal{#3}{rc}\or\equal{#3}{rb}}{}%
     {\PackageError{boxpage}{^^JInvalid argument "posb" for putbox %
                            (must be one of lt,lc,lb,ct,cc,cb,rt,rc,rb)}}
  \ifthenelse{\equal{#4}{t}\or\equal{#4}{c}\or\equal{#3}{b}\or\equal{#4}{s}\or%
              \equal{#4}{}}{}%
     {\PackageError{boxpage}{^^JInvalid argument "posi" for putbox %
                            (must be one of t, c, b, s)}}
%
%   -- set fbox margin and rule width and default auxiliary variables
%
  \setlength{\fboxsep}{\putbox@fsep}
  \setlength{\fboxrule}{\putbox@fwid}
  \setlength{\pb@frwidth}{\fboxrule+\fboxsep}
  \setlength{\pb@rboffset}{0.pt}
  \setcounter{pb@bpfr}{0}
  \pb@mpheighttrue
%
%   -- determine widths of box and minipage
%
  \ifthenelse{\equal{#7}{s}}%
    {\setlength{\pb@width}{\textwidth}%
     \ifthenelse{\equal{#2}{b}}%
       {\setlength{\pb@mpwidth}{\pb@width}}
       {\ifthenelse{\equal{#2}{c}}%
         {\setlength{\pb@mpwidth}{\pb@width-2\fboxsep}}%
         {\setlength{\pb@mpwidth}{\pb@width-2\pb@frwidth}}}}%
    {\ifthenelse{\equal{#7}{0}}%
       {\ifthenelse{\not\equal{#2}{p}}%
          {\settowidth{\pb@mpwidth}{#9}%
           \ifthenelse{\equal{#2}{b}}%
             {\setlength{\pb@width}{\pb@mpwidth}}%
             {\ifthenelse{\equal{#2}{c}}%
               {\setlength{\pb@width}{\pb@mpwidth+2\fboxsep}}%
               {\setlength{\pb@width}{\pb@mpwidth+2\pb@frwidth}}}}{}}%
       {\setlength{\pb@width}{\boxpict@scale*(#7)}%
        \ifthenelse{\equal{#2}{b}}%
          {\setlength{\pb@mpwidth}{\pb@width}}
          {\ifthenelse{\equal{#2}{c}}%
            {\setlength{\pb@mpwidth}{\pb@width-2\fboxsep}}%
            {\setlength{\pb@mpwidth}{\pb@width-2\pb@frwidth}}}}}%
  \ifthenelse{\equal{#2}{B}}%
    {\setlength{\pb@rboffset}{\pb@frwidth}
     \setcounter{pb@bpfr}%
                {1*\ratio{\pb@frwidth+\boxpict@scale/2}{\boxpict@scale}}}{}%
%
%   -- determine heights of box and minipage
%      (not needed if natural height is used)
%
  \ifthenelse{\equal{#8}{s}}%
    {\setlength{\pb@toth}{\textheight}%
     \ifthenelse{\equal{#2}{b}}%
       {\setlength{\pb@mptoth}{\pb@toth}}
       {\ifthenelse{\equal{#2}{c}}%
         {\setlength{\pb@mptoth}{\pb@toth-2\fboxsep}}%
         {\setlength{\pb@mptoth}{\pb@toth-2\pb@frwidth}}}}%
    {\ifthenelse{\equal{#8}{0}}%
       {\pb@mpheightfalse}%
       {\setlength{\pb@toth}{\boxpict@scale*(#8)}%
        \ifthenelse{\equal{#2}{b}}%
          {\setlength{\pb@mptoth}{\pb@toth}}
          {\ifthenelse{\equal{#2}{c}}%
            {\setlength{\pb@mptoth}{\pb@toth-2\fboxsep}}%
            {\setlength{\pb@mptoth}{\pb@toth-2\pb@frwidth}}}}}%
%
%   -- produce minipage box (plain [p] or box with frame space [B])
%
  \ifthenelse{\equal{#2}{b}\or\equal{#2}{B}}{%
    \ifpb@mpheight%
       \sbox{\pb@box}{\color{\putbox@ctxt}%
                      \begin{minipage}[t][\pb@mptoth][#4]{\pb@mpwidth}%
                      #9%
                      \end{minipage}}%
     \else%
       \sbox{\pb@box}{\color{\putbox@ctxt}%
                      \begin{minipage}{\pb@mpwidth}%
                      #9%
                      \end{minipage}}\fi%
  }{}%
%
%   -- produce framed minipage box [F]
%
  \ifthenelse{\equal{#2}{F}}{%
    \ifpb@mpheight%
       \sbox{\pb@box}{\color{\putbox@cfra}\fbox{\color{\putbox@ctxt}%
                      \begin{minipage}[t][\pb@mptoth][#4]{\pb@mpwidth}%
                      #9%
                      \end{minipage}}}%
     \else%
       \sbox{\pb@box}{\color{\putbox@cfra}\fbox{\color{\putbox@ctxt}%
                      \begin{minipage}{\pb@mpwidth}%
                      #9%
                      \end{minipage}}}\fi%
  }{}%
%
%   -- produce colored minipage box (using colorbox) [c]
%
  \ifthenelse{\equal{#2}{c}}{%
    \ifpb@mpheight%
       \sbox{\pb@box}{\colorbox{\putbox@cbkg}{\color{\putbox@ctxt}%
                      \begin{minipage}[t][\pb@mptoth][#4]{\pb@mpwidth}%
                      #9%
                      \end{minipage}}}%
     \else%
       \sbox{\pb@box}{\colorbox{\putbox@cbkg}{\color{\putbox@ctxt}%
                      \begin{minipage}{\pb@mpwidth}%
                      #9%
                      \end{minipage}}}\fi%
  }{}%
%
%   --  produce framed colored minipage box (using fcolorbox) [f]
%
  \ifthenelse{\equal{#2}{f}}{%
    \ifpb@mpheight%
       \sbox{\pb@box}{\fcolorbox{\putbox@cfra}{\putbox@cbkg}{%
                      \color{\putbox@ctxt}%
                      \begin{minipage}[t][\pb@mptoth][#4]{\pb@mpwidth}%
                      #9%
                      \end{minipage}}}%
     \else%
       \sbox{\pb@box}{\fcolorbox{\putbox@cfra}{\putbox@cbkg}{%
                      \color{\putbox@ctxt}%
                      \begin{minipage}{\pb@mpwidth}%
                      #9%
                      \end{minipage}}}\fi%
  }{}%
%
%   --  put postscript picture into box
%
  \ifthenelse{\equal{#2}{p}}{%
    \ifthenelse{\equal{#7}{0}\and\equal{#8}{0}}%
      {\sbox{\pb@box}{\psfig{#9}}}{}
    \ifthenelse{\not\equal{#7}{0}\and\equal{#8}{0}}%
      {\sbox{\pb@box}{\psfig{#9,width=\the\pb@width}}}{}
    \ifthenelse{\equal{#7}{0}\and\not\equal{#8}{0}}%
      {\sbox{\pb@box}{\psfig{#9,height=\the\pb@toth}}}{}
    \ifthenelse{\not\equal{#7}{0}\and\not\equal{#8}{0}}%
      {\sbox{\pb@box}{\psfig{#9,width=\the\pb@width,height=\the\pb@toth}}}{}%
  }{}%
%
%   --  calculate dimensions of resulting box
%
  \settowidth{\pb@width}{\usebox{\pb@box}}
  \settoheight{\pb@height}{\usebox{\pb@box}}
  \settodepth{\pb@depth}{\usebox{\pb@box}}
  \setlength{\pb@toth}{\pb@height+\pb@depth}
  \setcounter{pb@bpsw}{1*\ratio{\pb@width+\boxpict@scale/2}{\boxpict@scale}}
%
%   --  calculate raisebox parameters
%
  \ifthenelse{\equal{#3}{lt}\or\equal{#3}{ct}\or\equal{#3}{rt}}%
    {\setlength{\pb@rbheight}{-\pb@height-\pb@rboffset}}{}
  \ifthenelse{\equal{#3}{lc}\or\equal{#3}{cc}\or\equal{#3}{rc}}%
    {\setlength{\pb@rbheight}{-\pb@height/2+\pb@depth/2}}{}
  \ifthenelse{\equal{#3}{lb}\or\equal{#3}{cb}\or\equal{#3}{rb}}%
    {\setlength{\pb@rbheight}{\pb@depth+\pb@rboffset}}{}
%
%   --  calculate horizontal position in picture
%
  \ifthenelse{\equal{#5}{c}}%
    {\setcounter{pb@bpsx}{\value{boxpage@width}/2}}
    {\ifthenelse{\equal{#5}{-0}}%
      {\setcounter{pb@bpsx}{\value{boxpage@width}-\value{pb@bpfr}}}
      {\ifthenelse{\equal{#5}{0}}%
        {\setcounter{pb@bpsx}{\value{pb@bpfr}}}
        {\setcounter{pb@bpsx}{#5}
         \ifthenelse{\value{pb@bpsx}<0}%
           {\setcounter{pb@bpsx}{\value{boxpage@width}+\value{pb@bpsx}%
                                -\value{pb@bpfr}}}
           {\setcounter{pb@bpsx}{\value{pb@bpsx}+\value{pb@bpfr}}}%
     }}}
  \ifthenelse{\equal{#3}{ct}\or\equal{#3}{cc}\or\equal{#3}{cb}}%
    {\setcounter{pb@bpsx}{\value{pb@bpsx}-\value{pb@bpsw}/2}}{}
  \ifthenelse{\equal{#3}{rt}\or\equal{#3}{rc}\or\equal{#3}{rb}}%
    {\setcounter{pb@bpsx}{\value{pb@bpsx}-\value{pb@bpsw}}}{}
%
%   --  calculate vertical position in picture
%
  \ifthenelse{\equal{#6}{c}}%
    {\setcounter{pb@bpsy}{\value{boxpage@height}/2}}
    {\ifthenelse{\equal{#6}{0}}%
      {\setcounter{pb@bpsy}{\value{boxpage@height}}}
      {\ifthenelse{\equal{#6}{-0}}%
        {\setcounter{pb@bpsy}{0}}
        {\setcounter{pb@bpsy}{#6}
         \ifthenelse{\value{pb@bpsy}>0}%
           {\setcounter{pb@bpsy}{\value{boxpage@height}-\value{pb@bpsy}}}
           {\setcounter{pb@bpsy}{-\value{pb@bpsy}}}}}}
%
%   --  write out box
%
  \ifthenelse{\equal{#1}{0}}%
    {\put(\thepb@bpsx,\thepb@bpsy){\raisebox{\pb@rbheight}{\usebox{\pb@box}}}}%
    {\put(\thepb@bpsx,\thepb@bpsy){\begin{rotate}{#1}%
     \raisebox{\pb@rbheight}{\usebox{\pb@box}}\end{rotate}}}
}
%------------------------------------------------------------------------------
%       default page decoration and auxiliaries
%       ---------------------------------------
%       \boxpagedecoration    -> defines item(s) to be repeated on all
%                                normal and (if inserted by hand) on plain
%                                pages
%       \putboxpagedecoration -> put \boxpagedecoration at the bottom of 
%                                a picture box page
%       \footerleft           -> left-adjusted text in footer line
%       \footercenter         -> centered text in footer line
%       \footerright          -> right-adjusted text in footer line
%       \footerrulewidth      -> width of horizontal rule in default footer
%       \footerruledist       -> extra distance between horizontal rule 
%                                and text in default footer (can be negative)
%       \bpdtcolor            -> text color in boxpage decoration (b.d.)
%       \bpdbcolor            -> background color in b.d.
%       \bpdfcolor            -> frame color in b.d.
%       \bpdfsep              -> separation between frame and text in b.d.
%       \bpdfwid              -> frame width in b.d.
%       \bpdtype              -> type of box used for b.d. (b or f)
%       \bpdposb              -> position of reference point w.r.t. box in b.d.
%       \bpdposi              -> vertical position of text in box
%       \bpdposx              -> x position of box
%       \bpdposy              -> y position of box
%       \bpdwdth              -> width of box
%       \bpdhght              -> total height of box
%------------------------------------------------------------------------------
\newcommand{\footerleft}{Name}
\newcommand{\footercenter}{Talk}
\newcommand{\footerright}{\theslide}
\newcommand{\footerrulewidth}{1.5pt}
\newcommand{\footerruledist}{0.pt}
\newcommand{\bpdfsep}{0.pt}
\newcommand{\bpdfwid}{0.pt}
\newcommand{\bpdtcolor}{black}
\newcommand{\bpdbcolor}{white}
\newcommand{\bpdfcolor}{black}
\newcommand{\bpdtype}{b}
\newcommand{\bpdposb}{lb}
\newcommand{\bpdposi}{}
\newcommand{\bpdposx}{0}
\newcommand{\bpdposy}{-0}
\newcommand{\bpdwdth}{0}
\newcommand{\bpdhght}{0}
\newcommand{\boxpagedecoration}{%
  \raisebox{\depth}{%
  \begin{minipage}{\textwidth}
    \color{\bpdtcolor}\rule[\footerruledist]{\textwidth}{\footerrulewidth}
    \tiny\footerleft\hfill\footercenter\hfill\footerright
  \end{minipage}}%
}
\newcommand{\putboxpagedecoration}{%
  {\sbox{\bp@deco}{\boxpagedecoration}
   \putboxparloc{\bpdfsep}{\bpdfwid}%
                {\bpdfcolor}{\bpdbcolor}{\bpdtcolor}
   \putbox{\bpdtype}{\bpdposb}{\bpdposi}{\bpdposx}{\bpdposy}%
          {\bpdwdth}{\bpdhght}{\usebox{\bp@deco}}}%
}
%------------------------------------------------------------------------------
%       The \dividelengths command
%       --------------------------
%       to divide two lengths and return real number as string
%
%       \dividelengths{len1}{len2}{dig}
%
%               len1 = divisor
%               len2 = dividend
%               dig  = number of digits after decimal point
%
%       The result is stored in \divlen@res .
%------------------------------------------------------------------------------
\newcommand{\dividelengths}[3]{%
  \setlength{\divlen@divisor}{#1}
  \setlength{\divlen@dividend}{#2}
  \setcounter{divlen@rata}{1*\ratio{\divlen@divisor}{\divlen@dividend}}
  \addtolength{\divlen@divisor}{-\thedivlen@rata\divlen@dividend}
  \setcounter{divlen@cnt}{0}
  \setcounter{divlen@ratb}{0}
  \whiledo{\value{divlen@cnt}<#3}{%
    \multiply\divlen@divisor by 10
    \setcounter{divlen@ratt}{1*\ratio{\divlen@divisor}{\divlen@dividend}}
    \addtolength{\divlen@divisor}{-\thedivlen@ratt\divlen@dividend}
    \setcounter{divlen@ratb}{10*\value{divlen@ratb}+\value{divlen@ratt}}
    \stepcounter{divlen@cnt}
  }
  \multiply\divlen@divisor by 2
  \setcounter{divlen@ratt}{1*\ratio{\divlen@divisor}{\divlen@dividend}}
  \ifthenelse{\value{divlen@ratt}>1}{\stepcounter{divlen@ratb}}{}
  \renewcommand{\divlen@res}{\thedivlen@rata.\thedivlen@ratb}
}
%------------------------------------------------------------------------------
%       Definition of bpgrid (copied from grid in epic.sty)
%       ---------------------------------------------------
%       to generate vertical scale labels starting on top on left side,
%       and horizontal labels starting from right on bottom.
%------------------------------------------------------------------------------
\newcount\d@lta
\newdimen\@delta
\newdimen\@@delta
\newcount\@gridcnt
\newcount\@gridbot
\newcounter{@gridcall}
\setcounter{@gridcall}{0}
\def\bpgrid(#1,#2)(#3,#4){\@ifnextchar [{\@bpigrid(#1,#2)(#3,#4)}%
{\@bpigrid(#1,#2)(#3,#4)[@,@]}}

\long\def\@bpigrid(#1,#2)(#3,#4)[#5,#6]{%
\stepcounter{@gridcall}%
\makebox(#1,#2){%
\@delta=#1pt\@@delta=#3pt\divide\@delta \@@delta\d@lta=\@delta%
\advance\d@lta \@ne\relax%
\ifthenelse{\value{@gridcall}<7}{\message{bpgrid=\the\d@lta\space x}}%
{\ifthenelse{\value{@gridcall}=7}{\message{bpgrid: no further messages}}{}}%
%% copied the definition of \line(0,1){#2} for some efficiency!.
\multiput(0,0)(#3,0){\d@lta}{\hbox to\z@{\hskip -\@halfwidth \vrule
         \@width \@wholewidth \@height #2\unitlength \@depth \z@\hss}}%
\ifx#5@\relax\else%
\global\@gridcnt=#5%
\global\@gridbot=#1%
\multiput(0,#2)(#3,0){\d@lta}{%
\makebox(0,0)[b]{\number\@gridcnt\vspace{2mm}%
                 \global\advance\@gridcnt by #3
                 \global\advance\@gridbot by -#3}}%
\global\advance\@gridbot by #3
\global\advance\@gridcnt by -#3
\multiput(\@gridcnt,0)(-#3,0){\d@lta}{%
\makebox(0,0)[t]{\vrule\@width0.pt\@height2.2ex\@depth\z@\hss\kern0.5em
                 \number\@gridbot\global\advance\@gridbot by #3}}%
\fi%
\@delta=#2pt\@@delta=#4pt\divide\@delta \@@delta\d@lta=\@delta%
\advance\d@lta \@ne\relax%
\ifthenelse{\value{@gridcall}<7}{\message{\the\d@lta . }}{}%
%% copied the definition of \line(1,0){#1} for some efficiency!.
\multiput(0,#2)(0,-#4){\d@lta}{\vrule \@height \@halfwidth \@depth \@halfwidth
         \@width #1\unitlength}%
\ifx#6@\relax\else
\global\@gridcnt=#6%
\global\@gridbot=#2%
\multiput(0,#2)(0,-#4){\d@lta}{%
\makebox(0,0)[r]{\number\@gridcnt\ \global\advance\@gridcnt by #4%
                 \global\advance\@gridbot by -#4}}%
\global\advance\@gridbot by #4
\multiput(#1,\@gridbot)(0,#4){\d@lta}{%
\makebox(0,0)[l]{\ \number\@gridbot\global\advance\@gridbot by #4}}%
\fi}}
%------------------------------------------------------------------------------
%       Color definitions
%       -----------------
%       1) basic colors
%------------------------------------------------------------------------------
\providecommand{\white}{\color{white}}
\providecommand{\black}{\color{black}}
\providecommand{\red}{\color{red}}
\providecommand{\blue}{\color{blue}}
\providecommand{\green}{\color{green}}
\providecommand{\cyan}{\color{cyan}}
\providecommand{\magenta}{\color{magenta}}
\providecommand{\yellow}{\color{yellow}}
%------------------------------------------------------------------------------
%       2) shades of gray
%------------------------------------------------------------------------------
\definecolor{lightestgray}{gray}{0.95}
\providecommand{\lightestgray}{\color{lightestgray}}
\definecolor{lightergray}{gray}{0.9}
\providecommand{\lightergray}{\color{lightergray}}
\definecolor{lightgray}{gray}{0.8}
\providecommand{\lightgray}{\color{lightgray}}
\definecolor{lightmidgray}{gray}{0.7}
\providecommand{\lightmidgray}{\color{lightmidgray}}
\definecolor{midgray}{gray}{0.5}
\providecommand{\midgray}{\color{midgray}}
\definecolor{darkgray}{gray}{0.3}
\providecommand{\darkgray}{\color{darkgray}}
\definecolor{darkestgray}{gray}{0.15}
\providecommand{\darkestgray}{\color{darkestgray}}
%------------------------------------------------------------------------------
%       3) shades of red
%------------------------------------------------------------------------------
\definecolor{lightestred}{rgb}{1.0,0.97,0.97}
\providecommand{\lightestred}{\color{lightestred}}
\definecolor{lightred}{rgb}{1.0,0.9,0.8}
\providecommand{\lightred}{\color{lightred}}
\definecolor{darkred}{rgb}{0.9,0.,.2}
\providecommand{\darkred}{\color{darkred}}
%------------------------------------------------------------------------------
%       4) shades of blue
%------------------------------------------------------------------------------
\definecolor{lightestblue}{rgb}{0.95,0.975,1}
\providecommand{\lightestblue}{\color{lightestblue}}
\definecolor{lightblue}{rgb}{0.8,0.95,1}
\providecommand{\lightblue}{\color{lightblue}}
\definecolor{darkblue}{rgb}{0.,0.,.6}
\providecommand{\darkblue}{\color{darkblue}}
%------------------------------------------------------------------------------
%       5) shades of green
%------------------------------------------------------------------------------
\definecolor{lightestgreen}{rgb}{0.95,1.,0.95}
\providecommand{\lightestgreen}{\color{lightestgreen}}
\definecolor{lightgreen}{rgb}{0.8,1.,.8}
\providecommand{\lightgreen}{\color{lightgreen}}
\definecolor{darkgreen}{rgb}{0.,.6,0.1}
\providecommand{\darkgreen}{\color{darkgreen}}
%------------------------------------------------------------------------------
%       6) shades of cyan
%------------------------------------------------------------------------------
\definecolor{lightestcyan}{rgb}{0.95,1.0,1.0}
\providecommand{\lightestcyan}{\color{lightestcyan}}
\definecolor{lightcyan}{rgb}{0.9,1.0,1.0}
\providecommand{\lightcyan}{\color{lightcyan}}
\definecolor{darkcyan}{rgb}{0.,.6,.8}
\providecommand{\darkcyan}{\color{darkcyan}}
%------------------------------------------------------------------------------
%       7) shades of magenta
%------------------------------------------------------------------------------
\definecolor{lightestmagenta}{rgb}{1.0,0.975,1.0}
\providecommand{\lightestmagenta}{\color{lightestmagenta}}
\definecolor{lightmagenta}{rgb}{1.0,0.95,1.0}
\providecommand{\lightmagenta}{\color{lightmagenta}}
\definecolor{darkmagenta}{rgb}{0.6,0.,0.6}
\providecommand{\darkmagenta}{\color{darkmagenta}}
%------------------------------------------------------------------------------
%       8) shades of yellow
%------------------------------------------------------------------------------
\definecolor{lightyellow}{rgb}{1.0,1.0,0.9}
\providecommand{\lightyellow}{\color{lightyellow}}
\definecolor{darkyellow}{rgb}{0.8,0.8,0.5}
\providecommand{\darkyellow}{\color{darkyellow}}
%------------------------------------------------------------------------------
%       9) other colors
%------------------------------------------------------------------------------
\definecolor{money}{rgb}{0.4,0.8,0.}
\providecommand{\money}{\color{money}}
\definecolor{lightestpink}{rgb}{1.0,0.95,0.975}
\providecommand{\lightestpink}{\color{lightpink}}
\definecolor{lightpink}{rgb}{1.0,0.9,0.95}
\providecommand{\lightpink}{\color{lightpink}}
\definecolor{purple}{rgb}{0.9,0.0,0.8}
\providecommand{\purple}{\color{purple}}
\definecolor{orange}{rgb}{0.9,0.6,0.2}
\providecommand{\orange}{\color{orange}}
\definecolor{newgr}{rgb}{0.2,0.8,0.2}
\providecommand{\newgr}{\color{newgr}}
\definecolor{newbl}{rgb}{0.3,0.6,0.8}
\providecommand{\newbl}{\color{newbl}}
\definecolor{newor}{rgb}{1.0,0.6,0.}
\providecommand{\newor}{\color{newor}}
%------------------------------------------------------------------------------
%       boxpage initialization
%       ----------------------
%------------------------------------------------------------------------------
\putboxdefault
%------------------------------------------------------------------------------
%       Set page layout parameters
%       --------------------------
%       ... horizontal spacings
%------------------------------------------------------------------------------
\setlength{\hoffset}{-1.in}
\setlength{\evensidemargin}{2.cm-\boxpict@scale/4}
\setlength{\oddsidemargin}{2.cm-\boxpict@scale/4}
\setlength{\textwidth}{\paperwidth-\evensidemargin-\oddsidemargin}
%------------------------------------------------------------------------------
%       ... vertical spacings
%------------------------------------------------------------------------------
\setlength{\voffset}{-1.in}
\setlength{\topmargin}{2.cm}
\setlength{\headheight}{0.0cm}
\setlength{\headsep}{0.0cm}
\setlength{\footskip}{1.5cm-\boxpict@scale/2}
\setlength{\textheight}{\paperheight-\topmargin-\footskip}
%------------------------------------------------------------------------------
%       ... empty pages
%------------------------------------------------------------------------------
\pagestyle{empty}
\catcode`\@=12 % @ signs are no longer letters
%%
%% End of file `boxpage.sty'.
