import pickle
import random
from typing import List, Tuple

from accelerate import init_empty_weights
import torch

from model import ChameleonXLLMXConfig, ChameleonXLLMXForConditionalGeneration
from xllmx.data.data_reader import read_general
from xllmx.data.item_processor import ItemProcessorBase
from xllmx.solvers.finetune import FinetuneSolverBase


class ItemProcessor(ItemProcessorBase):
    def process_item(self, data_item: dict, training_mode=False) -> Tuple[List, List]:
        assert training_mode

        ori_item = data_item

        if "token" in data_item and "label" in data_item:
            data_item = data_item
        else:
            if "file" in data_item:
                if "s3://" in data_item["file"]:
                    data_item = pickle.load(read_general(data_item["file"]))
                else:
                    with open(data_item["file"], "rb") as f:
                        data_item = pickle.load(f)

        tokens = data_item["token"]
        labels = data_item["label"]
        assert len(tokens) == len(labels)

        if tokens[-2] == labels[-2] == 8196 and tokens.count(8196) == 1:  # image data
            # system prompt

            # "<s>Generate an image"
            assert tokens[:5] == [0, 22807, 16758, 16667, 19521], f"real: {tokens[:5]}"
            if "zl5.5" in ori_item["file"]:
                # "<s>Generate a high-quality photorealistic image or real photo"
                new_prefix = [
                    0,
                    22807,
                    16758,
                    16641,
                    17362,
                    16396,
                    29606,
                    18345,
                    63824,
                    18815,
                    19521,
                    16785,
                    17446,
                    21385,
                ]  # noqa
                tokens = new_prefix + tokens[5:]
                labels = [-100] * len(new_prefix) + labels[5:]
            elif "ideo" in ori_item["file"]:
                # "<s>Generate a high-quality synthetic image"
                new_prefix = [0, 22807, 16758, 16641, 17362, 16396, 29606, 35223, 19521]
                tokens = new_prefix + tokens[5:]
                labels = [-100] * len(new_prefix) + labels[5:]

            # for CFG
            if random.random() < 0.1:
                tokens = labels = [_ for _ in labels[:-1] if _ != -100]

        # if random.random() < 0.1:
        #     images = self._extract_images(tokens, labels)
        #     images = [_ for _ in images if _[1] == _[0]]  # only in the response part (to predict)
        #     if len(images) > 0:
        #         tokens, labels = random.choice(images)

        assert len(tokens) == len(labels)

        return tokens, labels

    def _extract_images(self, tokens, labels):

        image_start = 8197  # "<racm3:break>"
        image_end = 8196  # "<eoss>"

        result = []
        current_tokens = []
        current_labels = []

        for tok, lab in zip(tokens, labels):
            if tok == image_start:
                assert len(current_tokens) == 0
                # a new image
                current_tokens = [tok]
                current_labels = [lab]
            elif tok == image_end:
                # end of the current image
                current_tokens.append(tok)
                current_labels.append(lab)
                result.append((current_tokens, current_labels))
                current_tokens = []
                current_labels = []
            elif current_tokens:
                # current image is in construction
                current_tokens.append(tok)
                current_labels.append(lab)

        return result

    def predict_item_token_length(self, data_item: dict) -> int:
        if "token" in data_item:
            return len(data_item["token"])
        elif "len" in data_item:
            return data_item["len"]
        else:
            raise ValueError()


class Solver(FinetuneSolverBase):
    @classmethod
    def get_args_parser(cls):
        parser = super().get_args_parser()
        # task-specific parameters
        parser.add_argument("--max_seq_len", default=4096, type=int, help="max token length")
        parser.add_argument("--mask_image_logits", default=True)
        parser.add_argument("--unmask_image_logits", action="store_false", dest="mask_image_logits")
        parser.add_argument("--dropout", type=float, default=0.0)
        parser.add_argument("--z_loss_weight", type=float, default=0.0)
        parser.add_argument("--model_size", type=str, default="7B", choices=["7B", "34B"])
        return parser

    def _model_func(
        self,
        init_from: str,
    ) -> (ChameleonXLLMXForConditionalGeneration, None):

        # Only instantiate the model on rank0
        # Other ranks will receive the model weights from rank0 during FSDP wrapping (through `sync_module_states`)
        # See https://github.com/pytorch/pytorch/issues/105840
        if self.global_rank == 0:
            model = ChameleonXLLMXForConditionalGeneration.from_pretrained(
                init_from,
                max_position_embeddings=self.args.max_seq_len,
                mask_image_logits=self.args.mask_image_logits,
                dropout=self.args.dropout,
                z_loss_weight=self.args.z_loss_weight,
                torch_dtype=torch.bfloat16,
                device_map="cpu",
            )
        else:
            with init_empty_weights():
                config = ChameleonXLLMXConfig.from_pretrained(
                    init_from,
                    max_position_embeddings=self.args.max_seq_len,
                    mask_image_logits=self.args.mask_image_logits,
                    dropout=self.args.dropout,
                    z_loss_weight=self.args.z_loss_weight,
                    torch_dtype=torch.bfloat16,
                )
                model = ChameleonXLLMXForConditionalGeneration(config)

        del model.model.vqmodel

        return model, None

    def _item_processor_func(self) -> ItemProcessorBase:
        return ItemProcessor()

    def _make_and_save_starting_point(self, save_path: str) -> None:

        pretrained_name = {
            "7B": "Alpha-VLLM/Chameleon_7B_mGPT",
            "34B": "Alpha-VLLM/Chameleon_34B_mGPT",
        }[self.args.model_size]

        model = ChameleonXLLMXForConditionalGeneration.from_pretrained(
            pretrained_name,
            max_position_embeddings=self.args.max_seq_len,
            mask_image_logits=self.args.mask_image_logits,
            dropout=self.args.dropout,
            z_loss_weight=self.args.z_loss_weight,
            torch_dtype=torch.bfloat16,
            device_map="cpu",
        )

        image_tokens = model.model.vocabulary_mapping.image_tokens
        model.lm_head.weight.data[image_tokens] = torch.zeros_like(model.lm_head.weight.data[image_tokens])

        model.save_pretrained(save_path, max_shard_size="10GB")


if __name__ == "__main__":
    args = Solver.get_args_parser().parse_args()
    solver = Solver(args)
    solver.run()
