C++ Adventures: The Basics
Lately I’ve been learning C++ and coming from languages like Java and C#, it’s quite a transition. I’ve been meaning to get started with C++ for a while now, but never really got to it. In that time, there has been lots of talk to whereas the language would be a big step, because of the complicated aspects one doesn’t have in Java-like languages.
The biggest difference between C++ and Java can pretty much be described as one thing: memory management. The handling of available memory is almost completely abstracted away when using Java, you only have to worry about not overloading the device you’re developing on, which is rarely a problem when developing desktop applications. In C++, however, the allocation of memory is done for you, but releasing it is not.
Consider the following piece of code:
public void changeGear(int gear) { Gearbox gb = new Gearbox(); gb.setGear(gear); }
This is a simple piece of code, which would work flawlessly in Java-like languages. A new object of the type ‘Gearbox’ is created, a method from that object is called and the method is done. In C++, this method would cause severe problems in some cases. First of all, the object is created on the stack instead of the heap. If you want to know more about stack an heap, take a look at this post. Actually, this code wouldn’t run at all, but I’ll come back to that in a bit.
public void changeGear(int gear) { Gearbox* gb = new Gearbox(); gb->setGear(gear); }
Two things have changed: the type of the object is appended with an asterisk * and the method call is done through an arrow -> instead of a period.
What this does, is create a new object of type Gearbox on the heap and store a reference to that object on the stack, named 'gb’. In Java one would say “I have an object of type 'Gearbox’ called 'gb’”, but in C++, this would be “I have a reference called 'gb’ to an object of type 'Gearbox’. The variable named 'gb’ really is a number, namely the address of the piece of memory the object resides in. The asterisk indicates it is a reference to an object of type Gearbox instead of an object of type Gearbox. This is also called a 'Pointer’.
Because the 'gb’ variable no longer is the actual Gearbox object, we can’t use the period notation to call methods on it, it’s not an object after all. So what we can do to access the object’s members, is use the arrow -> notation. The third line of code tells the compiler to "get the object referenced by the 'gb’ variable and call it’s method named 'setGear’”. So where the period could be described as “get the following member of this object”, the arrow could be described as “get the following member of the object referenced by this variable”.
Earlier, I mentioned the code in the first snippet wouldn’t work at all C++. This is because the 'new’ keyword creates a new object on the stack and returns a reference to this object. As shown above, such a reference is represented by appending an asterisk to the type of the object (Gearbox*). The variable 'gb’ in the first snippet doesn’t have this asterisk (Gearbox). The syntax for creating an object on the stack looks like this:
public void changeGear(int gear) { Gearbox gb; gb.setGear(gear); }
This may seem quite unusual, as if we’re declaring an object without instantiating it. Still, the object really gets instantiated, assuming it has a constructor with no parameters (also known as a default constructor). If it does have one or more parameters, the code becomes like this:
public void changeGear(int gear) { Gearbox gb(6); gb.setGear(gear); }
So much for the first issue.
Secondly, the object stays in memory, even after the method is done. While Java releases any reserved memory within the current scope, at the end of the scope, C++ does not. We need to add a command to do that:
public void changeGear(int gear) { Gearbox* gb = new Gearbox(); gb->setGear(gear); delete gb; }
Now, the memory manager will release the bit of memory used for the Gearbox object.
When a piece of memory is 'released’, it becomes available for allocation at another point in your application’s execution. The memory itself is not modified, so it retains its data. The problem of this method is, that when the piece of memory is allocated a second time, there’s no way of telling what will be in there. Take a look at the following code:
public void enableWiper() { SoapReservoir* soapRes; }
In the above snippet, we’re enabling the wipers, to clean the windshield. Before we enable the wipers, we’ll check whether there’s enough soap in the reservoir. Now suppose we first declare the soapRes variable, before we’re going to assign it a value. In Java, the variable would have a value of null. In C++, you never know what soapRes’ value will be.
This is because the piece of memory reserved to contain the value of soapRes probably has some leftover data, which could have been another type like a float, or maybe even part of an object. To ensure the default value, the variable must be assigned.
public void enableWiper() { SoapReservoir* soapRes = NULL; }
This is useful for situations where you want to check whether an object is set or not:
public void turnOnRadio() { // Check if we even have a radio if(radio != NULL) { radio.turnOn(); } }
If the radio object hasn’t been assigned the NULL value on declaration, it will not be null, while the variable hasn’t been assigned. This will cause the above code to think there actually is a radio object and throw an exception when the radio object doesn’t appear to have a method called turnOn(), or alternatively, the application crashes.
That’s about it for the basics, next time I’ll talk about object orientation in C++. Going to sleep now :)